import Control.Arrow

-- Language Syntax

type Id        = String
type Primitive = Int -> Int
type BWrapper  = Int -> (Int,Int)
type AWrapper  = (Int,Int) -> Int
type Wrapper   = (Id,(BWrapper, AWrapper))

data Component = PC Primitive
               | BW Id BWrapper
               | AW Id AWrapper
               | CC Component Component
               | WC Id BWrapper Component AWrapper


data Direction = L | R deriving Show
type Transformation = Component -> Component
type LTrans = ([Direction],Transformation)

baseToView :: Component -> LTrans -> Component
baseToView c ([],trf) = trf c
baseToView (CC c1 c2) ((L:lt),trf) = (CC (baseToView c1 (lt,trf)) c2)
baseToView (CC c1 c2) ((R:lt),trf) = (CC c1 (baseToView c2 (lt,trf)))

-- Primitive Transformation Rules

shiftL :: Component -> Component
shiftL (CC c1  (CC c2 c3))   = (CC (CC c1 c2) c3)

shiftR :: Component -> Component
shiftR (CC (CC c1 c2) c3)    = (CC c1 (CC c2 c3))

-- Wrapping Transformation

type WTrans = ([Direction],Wrapper)

wrap :: Component -> Wrapper  -> Component
wrap (PC c)     (id,(before,after))  = WC id before (PC c) after
wrap (CC c1 c2) (id,(before,after))  = CC (CC (BW id before) c1) (CC c2 (AW id after)) 

wrapping :: Component -> WTrans -> Component
wrapping c ([],w)                = wrap c w 
wrapping (CC c1 c2) ((L:trf),w)  = (CC (wrapping c1 (trf,w)) c2)
wrapping (CC c1 c2) ((R:trf),w)  = (CC c1 (wrapping c2 (trf,w)))

-- Evaluation Function 
 
eval :: (Arrow cat) => Component ->  cat (Int,[(Id,Int)]) (Int,[(Id,Int)])
eval (PC c)                                     = first (arr c)
eval (BW id b)                                  = first (arr b) >>> arr (\((x,y),as) -> (x, (id,y):as))
eval (AW id a)                                  = arr (\ (x,as) -> ((x, getCtr id as), update id as)) >>> first (arr a)
eval (WC id b c a)                              = eval (BW id b) >>> eval c >>> eval (AW id a)
eval (CC c1 c2)                                 = eval c1 >>> eval c2


getCtr :: Id -> [(Id,Int)] -> Int
getCtr id ((id',v):as)
                  | id == id' = v
                  | otherwise = getCtr id as

update :: Id -> [(Id,Int)] -> [(Id,Int)]
update id ((id',v):as)
                  | id == id' = as
                  | otherwise = (id',v):(update id as) 

--------- Instantiation 

-- Primitive functions

f :: Int -> Int
f a = a + 3

g :: Int -> Int
g a = a*2
  
h :: Int -> Int
h a = a -2 

-- Wrappers

before1 :: Int -> (Int, Int)
before1 a = (a+1,5)

after1 :: (Int, Int) -> Int
after1 (a,b) = a*b

before2 :: Int -> (Int, Int)
before2 a = (a+2,6)


after2 :: (Int,Int) -> Int
after2 (a,b) = a+b

--- Example 1 : two crosscuting aspects 

base1   = (CC (CC (PC f) (PC g)) (PC h))

wBase1  = wrapping base1 ([L],("a",(before1,after1)))

view1   = baseToView wBase1 ([],shiftR)

final1  = wrapping view1 ([R],("b",(before2,after2)))

-- Example 2 : three crosscuting aspects 

base2 = CC (CC (CC (PC f) (PC g)) (PC h)) (CC (PC f) (PC h))

wBase2 = wrapping base2 ([L],("a",(before1,after1)))

view21 = baseToView wBase2 ([L,L],shiftL)

view22 = baseToView view21 ([L],shiftR)

view23 = baseToView view22 ([],shiftR)

view24 = baseToView view23 ([R],shiftL)

wView24 = wrapping view24 ([R,L],("b",(before2,after2)))

view25 = baseToView wView24 ([R,L,L],shiftL)

view26 = baseToView view25 ([R,L],shiftR)

view27 = baseToView view26 ([R],shiftR)

final2 = wrapping view27 ([R,R],("c",(before1,after1)))
