------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            FACL Version 1.1 Created by Abdelhakim Hannousse 29/04/2009                                                   --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
-- This version provides a precise implementation of formal aspect and component language (FACL), it has the following features:
--     1) it shows the base to view transformation as just one function call
--     2) before and after parts of wrappers are propagated as deeper as possible in the component structure with respect to the original view 
--     3) adding new wrappers does not affect the other already applied wrappers
--     4) no identifiers are used to show the dependency between before and after parts of wrappers
--     5) adapters are considered as functuions that make the control output of before wrappers on the bottom of all existing wires, and when an after wrapper will be inserted
--        a reverse function is used to make that wire in the second position to satisfy the after wrapper arrow.
--     6) LRC, FOT, and SPT binary composition operators are modeled following to arrows specifification 
--     7) current version works with components with different input and output wires number
--     8) Fst operator should be distribured in the part where the wrapper will be inserted, that ensures that the before and after parts of the wrapper are correctely propagated
--     9) Considering the Snd operator makes the number of input and output wires not sufficient to insert wrappers correctly, thus additional functions are used to capture the 
--        strucure of the component to be wrapped 
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

module FACL where 
 
import Control.Arrow
import System.IO

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            FACL Syntax Language                                                                                          --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

type Primitive       = String
type Transformation  = Component   -> Component
type Wrapper         = (String,String)
type WTrans          = ([Direction],Wrapper)
type LTrans          = ([Direction],Transformation)
type Type            = (Int,Int)

data Component       = Arr    Primitive   Type
                     | Fst    Component
                     | Snd    Component
                     | LOOP   Component
                     | LRC    Component   Component
                     | FOT    Component   Component
                     | SPT    Component   Component

data Direction       = L | R 

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            Primitive Transformation Rules                                                                                --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
 
shiftL                              :: Component  ->  Component
shiftL (LRC c1  (LRC c2 c3))         =  (LRC (LRC c1 c2) c3)
shiftL (FOT c1  (FOT c2 c3))         =  (FOT (FOT c1 c2) c3)
shiftL (SPT c1  (SPT c2 c3))         =  (SPT (SPT c1 c2) c3)

shiftR                               :: Component  ->  Component
shiftR (LRC (LRC c1 c2) c3)          =  (LRC c1 (LRC c2 c3))
shiftR (FOT (FOT c1 c2) c3)          =  (FOT c1 (FOT c2 c3))
shiftR (SPT (SPT c1 c2) c3)          =  (SPT c1 (SPT c2 c3))

fstDist                              :: Component  ->  Component
fstDist (Fst (LRC c1 c2))            =  LRC  (Fst c1)  (Fst c2)

fstFact                              :: Component  ->  Component
fstFact (LRC (Fst c1) (Fst c2))      =  Fst (LRC c1  c2) 

sndDist                              :: Component  ->  Component
sndDist (Snd (LRC c1 c2))            =  LRC (Snd c1)  (Snd c2) 

sndFact                              :: Component  ->  Component
sndFact (LRC (Snd c1) (Snd c2))      =  Snd (LRC c1  c2) 

setFst                               :: Component  ->  Component
setFst c                             =  Fst c

setSnd                               :: Component  ->  Component
setSnd c                             =  Snd c

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            Base To View Transformation                                                                                   --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

baseToView                         :: Component  ->  [LTrans]  ->  Component
baseToView c []                    =  c
baseToView c (t:ts)                =  baseToView (tstep c t) ts


tstep                              :: Component  ->  LTrans  ->  Component
tstep c ([],trf)                   =  trf c
tstep (LRC c1 c2) ((L:lt),trf)     =  (LRC (tstep c1 (lt,trf)) c2)
tstep (LRC c1 c2) ((R:lt),trf)     =  (LRC c1 (tstep c2 (lt,trf)))
tstep (FOT c1 c2) ((L:lt),trf)     =  (FOT (tstep c1 (lt,trf)) c2)
tstep (FOT c1 c2) ((R:lt),trf)     =  (FOT c1 (tstep c2 (lt,trf)))
tstep (SPT c1 c2) ((L:lt),trf)     =  (SPT (tstep c1 (lt,trf)) c2)
tstep (SPT c1 c2) ((R:lt),trf)     =  (SPT c1 (tstep c2 (lt,trf)))

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            Wrapping Function                                                                                             --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

wrapping                           :: Component  ->  WTrans  ->  Component
wrapping c ([],w)                  =  wrap c w
wrapping (LRC c1 c2) ((L:trf),w)   =  (LRC (wrapping c1 (trf,w)) c2)
wrapping (LRC c1 c2) ((R:trf),w)   =  (LRC c1 (wrapping c2 (trf,w)))
wrapping (FOT c1 c2) ((L:trf),w)   =  (FOT (wrapping c1 (trf,w)) c2)
wrapping (FOT c1 c2) ((R:trf),w)   =  (FOT c1 (wrapping c2 (trf,w)))
wrapping (SPT c1 c2) ((L:trf),w)   =  (SPT (wrapping c1 (trf,w)) c2)
wrapping (SPT c1 c2) ((R:trf),w)   =  (SPT c1 (wrapping c2 (trf,w)))


wrap                               :: Component  ->  Wrapper  ->  Component
wrap (LRC c1 c2) (before,after)    =  let (n1,n2) = (getPureInNumber c1, getPureOutNumber c2)
                                        in let c = bpropagate (Arr before (n1,n1+1)) c1
                                             in LRC c (apropagate (Arr after (n2+1,n2)) c2) 
wrap c (before,after)              =  let (n1,n2,ltrf)  = (getPureInNumber c, getPureOutNumber c, getCompStr c [])
                                        in LRC (LRC (setCompStr (Arr before (n1,n1+1)) ltrf) (setFst c)) (setCompStr (Arr after (n2+1,n2)) ltrf) 


bpropagate                         :: Component  ->  Component  ->  Component
bpropagate  c1 (LRC c2 c3)         =  let c' = bpropagate c1 c2
                                        in (LRC c' (setFst c3))
bpropagate  c1 c                    | getInNumber c == getInNumber c1    = LRC c1  (setFst c)
                                    | otherwise                          = let (n1,n2,ltrf) = (getInNumber c1, getInNumber c, getCompStr c []) 
                                                                             in LRC (LRC (setCompStr c1 ltrf) (genAd (n2-n1+2))) (setFst c)
 
apropagate                         :: Component  ->  Component  ->  Component
apropagate  c1 (LRC c2 c3)         =  LRC (setFst c2) (apropagate c1 c3) 
apropagate  c1 c                    | getOutNumber c == getInNumber c1 - 1 = LRC (setFst c) c1 
                                    | otherwise                            = let (n1,n2,ltrf) = (getOutNumber c, getInNumber c1, getCompStr c [])
                                                                               in LRC (setFst c) (LRC (rgenAd (n1-n2+3)) (setCompStr c1 ltrf))  

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            Adapter Generation Functions                                                                                  --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

genAd                              :: Int  ->  Component
genAd  n                            | n == 3    = Arr "swap" (3,3)
                                    | otherwise = LRC (setFst (genAd (n-1))) (Arr "swap" (n,n))

rgenAd                             :: Int  ->  Component
rgenAd n                            | n == 3    = Arr "swap" (3,3)
                                    | otherwise = LRC (Arr "swap" (n,n)) (setFst (rgenAd (n-1)))

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            Helper Functions                                                                                              --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

getInNumber                           :: Component  ->  Int
getInNumber (Arr  c (nin,nout))        =  nin
getInNumber (Fst  c)                   =  1 + getInNumber c
getInNumber (Snd  c)                   =  1 + getInNumber c
getInNumber (LRC  c1 c2)               =  getInNumber c1
getInNumber (FOT  c1 c2)               =  getInNumber c1
getInNumber (SPT  c1 c2)               =  getInNumber c1 + getInNumber c2
getInNumber (LOOP c)                   =  getInNumber c  - 1

getPureInNumber                        :: Component  ->  Int
getPureInNumber (Arr  c (nin,nout))    =  nin
getPureInNumber (Fst  c)               =  getPureInNumber c
getPureInNumber (Snd  c)               =  getPureInNumber c
getPureInNumber (LRC  c1 c2)           =  getPureInNumber c1
getPureInNumber (FOT  c1 c2)           =  getPureInNumber c1
getPureInNumber (SPT  c1 c2)           =  getPureInNumber c1 + getPureInNumber c2
getPureInNumber (LOOP c)               =  getPureInNumber c  - 1

getOutNumber                           :: Component  ->  Int
getOutNumber (Arr  c (ninp,nout))      =  nout
getOutNumber (Fst  c)                  =  1 + getOutNumber c
getOutNumber (Snd  c)                  =  1 + getOutNumber c
getOutNumber (LRC  c1 c2)              =  getOutNumber c2
getOutNumber (FOT  c1 c2)              =  getOutNumber c1 + getOutNumber c2
getOutNumber (SPT  c1 c2)              =  getOutNumber c1 + getOutNumber c2
getOutNumber (LOOP c)                  =  getOutNumber c  - 1

getPureOutNumber                       :: Component  ->  Int
getPureOutNumber (Arr  c (ninp,nout))  =  nout
getPureOutNumber (Fst  c)              =  getPureOutNumber c
getPureOutNumber (Snd  c)              =  getPureOutNumber c
getPureOutNumber (LRC  c1 c2)          =  getPureOutNumber c2
getPureOutNumber (FOT  c1 c2)          =  getPureOutNumber c1 + getPureOutNumber c2 
getPureOutNumber (SPT  c1 c2)          =  getPureOutNumber c1 + getPureOutNumber c2 
getPureOutNumber (LOOP c)              =  getPureOutNumber c - 1

getCompStr                             :: Component  ->  [Transformation]  ->  [Transformation]
getCompStr   (Fst c)          ltrf     =  getCompStr c (setFst:ltrf)
getCompStr   (Snd c)          ltrf     =  getCompStr c (setSnd:ltrf)
getCompStr   c                ltrf     =  ltrf

setCompStr                             :: Component  ->  [Transformation]  ->  Component
setCompStr   c                []       =  c
setCompStr   c              (t:ltrf)   =  setCompStr (t c) ltrf

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            FACL To Arrows Transformation                                                                                 --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

genArr c                           = writeFile "/Users/hakim/Documents/Interpreters/Arrows/out.hs" 
                                               ("module Out where\n\n"++
                                                "import Lib\n"++ 
                                                "import Config\n"++ 
                                                "import Control.Arrow\n\n\n"++ 
                                                "a :: (Arrow cat, ArrowLoop cat) => cat ("++ genType (getInNumber c) ++")  ("++ genType (getOutNumber c) ++")\n"++ 
                                                "a = "++ (eval c))
 
eval                               :: Component  ->  String 
eval (Arr  c   (n1,n2))            =  "arr "++c
eval (Fst  c)                      =  "first  ("++eval c++")"
eval (Snd  c)                      =  "second ("++eval c++")"
eval (LRC  c1 c2)                  =  "("++eval c1++" >>> "++eval c2++")"
eval (FOT  c1 c2)                  =  "("++eval c1++" &&& "++eval c2++")"
eval (SPT  c1 c2)                  =  "("++eval c1++" *** "++eval c2++")"
eval (LOOP c)                      =  "loop ("++eval c++")"

genType                           ::  Int  ->  String
genType                         n  |  n == 1     = "Int"
                                   |  otherwise  = "Int," ++ genType (n-1)

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--                                                            Test Examples                                                                                                 --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--       Examples                                                                                                   Test Resuls                 Description                 --
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------     
base0   = (LRC (LRC (Arr "f" (1,1)) (Arr "g" (1,1))) (Arr "h" (1,1)))
final0  = wrapping     base1     ([R],("before1","after1"))                                                         -- in = 3 => out = 55       wrapping a primitive component
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base1   = (LRC (LRC (Arr "f" (1,1)) (Arr "g" (1,1))) (Arr "h" (1,1)))
wBase1  = wrapping    base1     ([L],("before1","after1"))
view1   = baseToView  wBase1    [([],shiftR)]
final1  = wrapping    view1     ([R],("before2","after2"))                                                          -- in = 3 => out = 94       two interleaving aspects
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base2   = LRC (LRC (LRC (Arr "f" (1,1)) (Arr "g" (1,1))) (Arr "h" (1,1))) (LRC (Arr "f" (1,1)) (Arr "h" (1,1)))
wBase2  = wrapping    base2     ([L],("before1","after1")) 
view22  = baseToView  wBase2    [([L],shiftR),([],shiftR),([R],shiftL)]
wView22 = wrapping    view22    ([R,L],("before2","after2")) 
view23  = baseToView  wView22   [([R,L],shiftR),([R],shiftR),([R,R,L,L],fstDist)]
final2  = wrapping    view23    ([R,R],("before1","after1"))                                                        -- in = 5 => out = 560      three interleaving aspects 
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base3   = LRC (LRC (LRC (Arr "f" (1,1)) (Arr "g" (1,1))) (Arr "h" (1,1))) (LRC (Arr "f" (1,1)) (Arr "h" (1,1)))
view31  = baseToView  base3     [([],shiftL)]
wView31 = wrapping    view31    ([L],("before1","after1")) 
view32  = baseToView  wView31   [([L,L],shiftR),([L],shiftR),([],shiftR)]
wView32 = wrapping    view32    ([R],("before2","after2"))                                                                                        
view33  = baseToView  wView32   [([R,L],shiftR),([R,L,R,R], fstDist),([R,L,R],shiftL)]                             --                            three applied aspects with
final3  = wrapping    view33    ([R,L,R,L],("before1","after1"))                                                   -- in = 5 => out = 604        only two interleaving aspects
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base4   = LRC (LRC (Arr "f" (1,1)) (Arr "g" (1,1))) (LRC (Arr "f" (1,1)) (Arr "h" (1,1)))
wBase4  = wrapping    base4    ([],("before1","after1")) 
view41  = baseToView  wBase4   [([],shiftL),([L],shiftR)]                                                         
final4  = wrapping    view41   ([L,R],("before2","after2"))                                                        -- in = 5 => out = 145        two interleaving aspects
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base5   = LRC (LRC (LRC (Arr "f" (1,1)) (Arr "g" (1,1))) (Arr "h" (1,1))) (LRC (Arr "f" (1,1)) (Arr "h" (1,1)))
wBase5  = wrapping    base5    ([L,L],("before1","after1"))
view51  = baseToView  wBase5   [([],shiftR)]     
wView51 = wrapping    view51   ([R],("before1","after1"))
view52  = baseToView  wView51  [([],shiftR),([R],shiftL)]                                                                                    
final5  = wrapping    view52   ([R,L],("before2","after2"))                                                        -- in = 3 => out = 480         three interleaving aspects   
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base6   = (LRC (LRC (Arr "f'" (2,3)) (Arr "g'" (3,2))) (Arr "h'" (2,2)))
wBase6  = wrapping    base6    ([L],("before3","after3"))
view61  = baseToView  wBase6   [([],shiftR)]                                                                       --                             two interleaving aspects with
final6  = wrapping    view61   ([R],("before4","after4"))                                                          -- in =(1,2) => out =(120,15)  different inputs and outputs 
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base7   = LRC (LRC (LRC (Arr "f'" (2,3)) (Arr "g'" (3,2))) (Arr "h'" (2,2))) (LRC (Arr "f'" (2,3)) (Arr "h'" (2,2)))
wBase7  = wrapping    base7    ([L],("before3","after3")) 
view71  = baseToView  wBase7   [([L],shiftR),([],shiftR),([R],shiftL)]
wView71 = wrapping    view71   ([R,L],("before4","after5")) 
view72  = baseToView  wView71  [([R,L],shiftR),([R],shiftR),([R,R,L,L],fstDist)]                                                       --         three interleaving aspects with
final7  = wrapping    view72   ([R,R],("before3","after3"))                                                        -- in=(1,2) => out=(3141,3927) different inputs and outputs
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base8   = (LRC (LRC (Arr "f1" (2,2)) (FOT (Arr "g1" (2,1))  (Arr "g2" (2,1)))) (Arr "h1" (2,1)))
wBase8  = wrapping    base8    ([L],("before6","after6"))
view8   = baseToView  wBase8   [([],shiftR)]                                                                       --                             two interleaving aspects with  
final8  = wrapping    view8    ([R],("before7","after7"))                                                          -- in = (1,2) => out = 265     FOT (&&&) composition operator
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base9   = (LRC (LRC (Arr "f9" (2,4)) (SPT (Arr "g1" (2,1))  (Arr "g2" (2,1)))) (Arr "h1" (2,1)))
wBase9  = wrapping    base9    ([L],("before6","after6"))
view9   = baseToView  wBase9   [([],shiftR)]                                                                       -- in = (1,2) => out = 128     two interleaving aspects with
final9  = wrapping    view9    ([R],("before9","after9"))                                                          -- in = (1,2) => out = 488     SPT (***) composition operator
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base10  = (LRC (LRC (Arr "f9" (2,4)) (SPT (Arr "g1" (2,1))  (Arr "g2" (2,1)))) (Arr "h1" (2,1)))                   --                             wrapping a part of the SPT   
final10 = wrapping    base10   ([L,R,L],("before10","after10"))                                                    -- in = (1,2) => out = 58      (***) structure
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
base11  = (LRC (Arr "f" (1,1))  (LOOP (Arr "f11" (2,2))))                                                          -- in = 1  => out = 8          an example with loop operator 
final11 = wrapping    base11   ([R],("before1","after1"))                                                          -- in = 3  => out = 70
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------