
import List

-- Boolean Expressions        

data BExp = 
    And BExp BExp 
  | Or  BExp BExp 
  | Not BExp 
  | Inf AExp AExp 
  | Sup AExp AExp 
  | Eqq AExp AExp 
  | BVl Bool 
  deriving Show

expB2 :: BExp
expB2 = Eqq (IVl 7) exp1

evalBExp :: BExp -> State -> Stack -> LProcd -> Global -> Args -> (Stack, Bool)
evalBExp (BVl e)       s st lps gl args = (st,e)
evalBExp (And e1 e2)   s st lps gl args = let (st1,b1) = evalBExp e1 s st lps gl args in 
                                              let (st2,b2) = evalBExp e2 s st1 lps gl args in (st2,b1 && b2)
evalBExp (Or e1 e2)    s st lps gl args = let (st1,b1) = evalBExp e1 s st lps gl args in 
                                              let (st2,b2) = evalBExp e2 s st1 lps gl args in (st2,b1 || b2)
evalBExp (Not e)       s st lps gl args = let (st1, b1) = evalBExp e s st lps gl args in (st1, not b1)
evalBExp (Inf e1 e2)   s st lps gl args = inf (evalAExp e1 s (evalAExp e2 s st lps gl args) lps gl args)
evalBExp (Sup e1 e2)   s st lps gl args = sup (evalAExp e1 s (evalAExp e2 s st lps gl args) lps gl args)
evalBExp (Eqq e1 e2)   s st lps gl args = eqq (evalAExp e1 s (evalAExp e2 s st lps gl args) lps gl args)

inf(v1:v2:vs) = (vs,(v1 < v2))
sup(v1:v2:vs) = (vs,(v1 > v2))
eqq(v1:v2:vs) = (vs,(v1 == v2))

-- Arithmetic Expressions

data AExp = 
    Add AExp AExp 
  | Sub AExp AExp 
  | Mul AExp AExp 
  | Div AExp AExp 
  | IVl Int
  | ID  String 
  | Call Id [AExp] 
  deriving (Show, Eq)

exp1 :: AExp
exp1 = Add (IVl 1) (Mul (IVl 2) (IVl 3))

type Stack = [Int]
 
evalAExp :: AExp -> State -> Stack -> LProcd -> Global -> Args -> Stack
evalAExp (IVl  i)      s st lps gl args = i:st
evalAExp (ID   id)     s st lps gl args = (get s id):st
evalAExp (Add  e1 e2)  s st lps gl args = add (evalAExp e2 s (evalAExp e1 s st lps gl args) lps gl args)
evalAExp (Sub  e1 e2)  s st lps gl args = sub (evalAExp e2 s (evalAExp e1 s st lps gl args) lps gl args)
evalAExp (Mul  e1 e2)  s st lps gl args = mul (evalAExp e2 s (evalAExp e1 s st lps gl args) lps gl args)
evalAExp (Div  e1 e2)  s st lps gl args = di  (evalAExp e2 s (evalAExp e1 s st lps gl args) lps gl args)
evalAExp (Call id prs) s st lps gl args = let (stack, _, _) = evalProcCall id lps ([s]++gl) ([prs]++args) st
                                          in stack

evalProcCall :: Id -> LProcd -> Global -> Args -> Stack -> (Stack, Global, Args)
evalProcCall id lps gl args st = let (_,(prm,stmt)) = matchProcd id (head args) lps
                                 in let (_,stack, global, arg) = evalStmt stmt (getLocalContext (head gl) (head args) prm st lps gl args) st lps gl args
                                    in (stack, global, arg)  

getLocalContext :: State -> [AExp] -> [Id] -> Stack -> LProcd -> Global -> Args -> State
getLocalContext vars vls prms st lps gl args = (rgetFst vars ((getFst vars) \\ prms)) ++ (mapPrm prms vls vars st lps gl args)

mapPrm :: [Id] -> [AExp] -> State -> Stack -> LProcd -> Global -> Args -> [Var]
mapPrm lid lint s st lps gl args = if (length lid == 0) then [] 
                           else (head lid, head (evalAExp (head lint) s st lps gl args)):(mapPrm (tail lid) (tail lint) s st lps gl args)

getStack :: (Stack, Global, Args) -> Stack
getStack (st, _, _)= st

add  (v1:v2:vs) = (v1 + v2):vs
sub  (v1:v2:vs) = (v2 - v1):vs
mul  (v1:v2:vs) = (v1 * v2):vs
di   (v1:v2:vs) = (div v2  v1):vs

-- Statements

type Id = String

type Var = (Id,Int)


type State = [Var] 

type Args   = [[AExp]]
type Global = [[Var]]

write :: State -> Id -> Int -> State
write s id v = (id,v):(delete (id,(get s id)) s)

get :: State -> Id -> Int
get ((id,v):s) id2 
  | id==id2   = v
  | otherwise = get s id2

data Stmt = 
    DecVar Id 
  | Assign Id AExp
  | AssignCal Id Stmt 
  | Sq Stmt Stmt 
  | If BExp Stmt Stmt
  | While BExp Stmt
  | Return AExp
  deriving Show


evalStmt :: Stmt -> State -> Stack -> LProcd -> Global -> Args -> (State, Stack, Global, Args)
evalStmt (DecVar id)               state stack procs gl args = (write state id 0, stack, gl, args)
evalStmt (Assign id e)             vars st procs gl args = let v:st'= evalAExp e vars st procs gl args
                                                           in (write vars id v, st', gl, args)
evalStmt (Sq s1 s2)                vars st procs gl args = let (vars', st', _, _)= evalStmt s1 vars st procs gl args
                                                           in evalStmt s2 vars' st' procs gl args
evalStmt (If a b c)                vars st procs gl args = if (snd(evalBExp a vars st procs gl args)) then evalStmt b vars st procs gl args
                                                           else evalStmt c vars st procs gl args
evalStmt (While a b)               vars st procs gl args = let (_,v) = evalBExp a vars st procs gl args
                                                           in if not(v) then (vars, st, gl, args) 
                                                              else let (vars',st', _ , _) = evalStmt b vars st procs gl args
                                                                   in evalStmt (While a b) vars' st' procs gl args
evalStmt (Return e)                vars st procs gl args = let stack = evalAExp e vars st procs gl args
                                                               in (head gl,stack, (getGl gl), (getArg args))

getGl :: Global -> Global
getGl [] = []
getGl (g:gl) = gl

getArg :: Args -> Args 
getArg [] = []
getArg (a:args) = args

body (_,(_,stmt)) = stmt

matchProcd :: Id -> [AExp] -> LProcd -> (Id,([Id],Stmt))
matchProcd id prm (p@(id',(prm',body)):lps)  
    = if (id'==id) && (length prm'==length prm) then p
      else matchProcd id prm lps

-- Procedures

type Params = [Var]
data Procd  = Pro Id Params Stmt deriving Show

getFst :: [Var] -> [Id]
getFst vs = map fst vs 

rgetFst :: [Var] -> [Id] -> [Var]
rgetFst vs ids = filter (\(id,val) -> id `elem` ids) vs

-- Programs 

type Param    = [Id]

type Local    = [Id]

type LProcd   = [(Id,([Id],Stmt))]

data DecVar  = Var Id deriving Show
type DecVars = [DecVar]

evalDecVars :: DecVars -> State
evalDecVars [] = []
evalDecVars ((Var a):dvs) = (a,0):(evalDecVars dvs)
 
data DecProc  = Proc Id Param Stmt  deriving Show
type DecProcd = [DecProc]

evalDecProcd :: DecProcd -> LProcd
evalDecProcd ((Proc id prm stmt):dprcs) = (id,(prm,stmt)):(evalDecProcd dprcs)
 
data Kernel   = 
      Prog Stmt 
      deriving Show

evalKernel :: Kernel -> State -> Stack -> LProcd -> Global-> Args -> (State, Stack, Global, Args)
evalKernel (Prog a) state stack lpcs global args = evalStmt a state stack lpcs global args

-- Program with procedure declaration

data Program  = P Id DecVars DecProcd Kernel deriving Show

evalProgram :: Program -> (State, Stack, Global, Args)
evalProgram (P id v p k) = evalKernel k (evalDecVars v) [] (evalDecProcd p) [] []

-- Printer

aexp2str :: AExp -> String
aexp2str (ID id)   = id
aexp2str (IVl v)   = show v
aexp2str (Add a b) = aexp2str(a)++"+"++aexp2str(b)
aexp2str (Sub a b) = aexp2str(a)++"-"++aexp2str(b)
aexp2str (Mul a b) = aexp2str(a)++"*"++aexp2str(b)
aexp2str (Div a b) = aexp2str(a)++"/"++aexp2str(b)
aexp2str (Call id prms) = id++"("++list2str(prms)++")"

bexp2str :: BExp -> String
bexp2str (BVl a)     = show a
bexp2str (And e1 e2) = bexp2str(e1)++"&&"++bexp2str(e2)
bexp2str (Or e1 e2)  = bexp2str(e1)++"||"++bexp2str(e2)
bexp2str (Not e)     = "!"++bexp2str(e)
bexp2str (Inf e1 e2) = aexp2str(e1)++"<"++aexp2str(e2)
bexp2str (Sup e1 e2) = aexp2str(e1)++">"++aexp2str(e2)
bexp2str (Eqq e1 e2) = aexp2str(e1)++"=="++aexp2str(e2)

stmt2str :: Stmt -> String
stmt2str (DecVar id)     = "Int "++id
stmt2str (Assign id v)   = id++"="++aexp2str(v)
stmt2str (Sq a b)        = stmt2str(a)++";"++stmt2str(b)
stmt2str (If a b c)      = "if("++bexp2str(a)++") {"++stmt2str(b)++";} else {"++stmt2str(c)++"; }"
stmt2str (While a b )    = "while ("++bexp2str(a)++") {"++stmt2str(b)++"; }"
stmt2str (Return e)     =  "return "++aexp2str(e)

program2str :: Program -> String
program2str (P id vd pd ker) = "class "++id++" {"++vdecls2str(vd)++pdecls2str(pd)++"public static void main(String[] args){"++kernel2str(ker)++";}}"

kernel2str :: Kernel -> String
kernel2str (Prog sts) = stmt2str sts

vdecls2str :: DecVars -> String
vdecls2str [] =""
vdecls2str ((Var a):dvs) = "static int "++a++";"++vdecls2str(dvs)

pdecls2str :: DecProcd -> String
pdecls2str [] = ""
pdecls2str ((Proc id prm sts):ldprcs) = "static void "++id++"("++vlist2str(prm)++")"++"{ "++stmt2str(sts)++"}"++pdecls2str(ldprcs)

list2str :: [AExp] -> String
list2str l = if ((length l)==0) then " " else if ((length l)==1) then aexp2str(head l) else (aexp2str(head l))++","++list2str(tail l)

vlist2str :: [Id] -> String
vlist2str l = if ((length l)==0) then " " else if ((length l)==1) then "int "++(head l) else "int "++(head l)++","++vlist2str(tail l)

--- Test examples

-- Factorial imperative program

prog1 :: Program
prog1 =
      P "factorial"
      ([(Var "k")])                                  
      ([(Proc "fact" ["n"]
                 (Sq (Assign "m" (IVl 1))
                     (Sq (While (Sup (ID "n") (IVl 1))
                            (Sq (Assign "m" (Mul (ID "m") (ID "n")))
                                (Assign "n" (Sub (ID "n") (IVl 1)))                      
                            )
                         )
                         (Return (ID "m"))
                     )
                 )
        )])
      (Prog (Assign "k" (Call "fact" [(IVl 7)])))

-- Factorial recursive program

prog2 :: Program
prog2 =
      P "factorial"
      ([(Var "n")])                                  
      ([(Proc "fact" ["n"]
                 (If (Eqq (ID "n") (IVl 1)) 
                     (Return (IVl 1)) 
                     (Return (Mul (ID "n") (Call "fact" [(Sub (ID "n") (IVl 1))]))))                
      )]) 
      (Prog (Assign "n" (Call "fact" [(IVl 5)])))

-- Fibonacci recursive program

prog3 :: Program
prog3 =
      P "fibonacci"
      ([(Var "n")])                                  
      ([(Proc "fib" ["n"]
                 (If (Eqq (ID "n") (IVl 0)) 
                         (Return (IVl 0))
                         (If (Eqq (ID "n") (IVl 1))
                             (Return (IVl 1))
                             (Return (Add (Call "fib" [(Sub (ID "n") (IVl 1))]) (Call "fib" [(Sub (ID "n") (IVl 2))])))
                         )
                 )
      )]) 
      (Prog (Assign "n" (Call "fib" [(IVl 11)])))


-- Greatest Common divisor

prog4 :: Program
prog4 =
      P "greteastCommonDivisor"
      ([(Var "n")])                                  
      ([(Proc "reminder" ["x","y"]
                 (Return (Sub (ID "x") (Mul (ID "y") (Div (ID "x") (ID "y")))))
        ),
        (Proc "gcd" ["x", "y"]
                 (If (Eqq (ID "y") (IVl 0)) 
                     (Return (ID "x")) 
                     (Return (Call "gcd" [(ID "y"),(Call "reminder" [(ID "x"),(ID "y")])])))                
      )]) 
      (Prog (Assign "n" (Call "gcd" [(IVl 259),(IVl 111)])))

-- Towers of Hanoi

prog5 :: Program
prog5 =
      P "towersOfHanoi"
      ([(Var "n")])                                  
      ([(Proc "hanoi" ["n"]
                 (If (Eqq (ID "n") (IVl 1)) 
                         (Return (IVl 1))
                         (If (Sup (ID "n") (IVl 1))
                             (Return (Add (Mul (IVl 2) (Call "hanoi" [(Sub (ID "n") (IVl 1))])) (IVl 1)))
                             (Return (IVl 0))
                         )
                 )
      )]) 
      (Prog (Assign "n" (Call "hanoi" [(IVl 4)])))

 
