module Oo where
import List

---- Language Syntax

--- primitive types

type Id    = String       
type Type  = String

--- expressions

data Exp = New  Type  [Exp]
         | Get  Exp   Id
         | Cast Type  Exp
         | Call Exp   Id  [Exp]
         | This Id
         | ID Id
         | Null
         deriving (Show, Eq)

--- method and constructor bodies

type MethBody = Exp 

data ConsBody = Super [Id] [Id] [Id] deriving (Show, Eq)

--- class members

data Field  = Fd Type Id deriving (Show, Eq) 
data Method = Meth Type Id [(Type,Id)] MethBody deriving (Show, Eq)
data Const  = Cons Id      [(Type,Id)] ConsBody deriving (Show, Eq)
 
--- class declaration

data CL = Class Id Id Const [Field] [Method] deriving (Show, Eq)

--- object oriented program

type Prg = [CL]
data Program = Prog [CL] Exp deriving (Show, Eq)

----- Language Semantics

--- semantic domains

type FIeld = (Type, Id, Exp)
type Methd = (Type, Id, [(Type, Id)], MethBody)
type CLass = (Id, Id, ([(Type, Id)], ConsBody),[FIeld],[Methd])
type Stack = [CLass]
 
--- auxiliary functions

classLookup :: Id -> Stack -> CLass
classLookup id1 ((id2,id3,(fs,cs),fss,ms):st)
         | id1 == id2 = (id2,id3,(fs,cs),fss,ms) 
         | otherwise  = classLookup id1 st  

fieldLookup :: Id -> Stack -> [FIeld]
fieldLookup id st
         | id == "Object" = []
         | otherwise       = let c = classLookup id st
                             in unionBy (\a b -> getFieldName(a) == getFieldName(b)) (getClassFieldList c)  (fieldLookup (getSuperClassName c) st)  

methodLookup :: Id -> Stack -> [Methd]
methodLookup id st
         | id == "Object" = []
         | otherwise       =  unionBy (\a b -> (getMethodType(a) == getMethodType(b)) && (getMethodName(a) == getMethodName(b)) && (matchFieldList (getFieldList(a))  (getFieldList(b)))) 
                                      (getClassMethodList(classLookup id st))  
                                      (methodLookup (getSuperClassName(classLookup id  st)) st)  

matchFieldList :: [(Type, Id)] -> [(Type, Id)] -> Bool
matchFieldList fs1 fs2
            | fs1 == [] && fs2 == []  = True
            | (length fs1 == length fs2) && (fst(head fs1) == fst(head fs2)) = matchFieldList (tail fs1) (tail fs2) 
            | otherwise               = False
       
getFieldList :: Methd -> [(Type, Id)]
getFieldList (_,_,fs,_) = fs

getFieldNameList :: [FIeld] -> [Id]
getFieldNameList [] = []
getFieldNameList (f:fs) = (getFieldName f):(getFieldNameList fs) 

getFieldName :: (Type, Id, Exp) -> Id
getFieldName (_, id, _) = id

getFieldType :: (Type, Id, Exp) -> Type
getFieldType (t, _, _) = t

getField :: Id -> [FIeld] -> FIeld
getField id  fs
       | fs == []   = ("null",id,Null)
       | id == getFieldName (head fs)  = head fs
       | otherwise  = getField id (tail fs)   

getMethod :: Id -> [Methd] -> Methd
getMethod m (m':ms) 
       | m == (getMethodName m') = m'
       | otherwise = getMethod m ms

getMethodType :: Methd -> Type
getMethodType (t,_,_,_) = t

getMethodName :: Methd -> Id
getMethodName (_,id,_,_) = id 

getClassFieldList :: CLass -> [FIeld]
getClassFieldList (_,_,(_,_),fs,_) = fs

getClassMethodList :: CLass -> [Methd]
getClassMethodList (_,_,(_,_),_,ms) = ms

getSuperClassName :: CLass -> Id
getSuperClassName (_,id,(_,_),_,_) = id

getSuperClassList :: Type -> Stack -> [Type]
getSuperClassList t st 
            | t == "Object" = ["Object"]
            | otherwise    = (getSuperClassList (getSuperClassName (classLookup t st)) st)++[t]

--- evaluation functions

evalProgram :: Program -> Exp
evalProgram (Prog cls e) = let st = evalClsDecls cls []
                           in let (e',t',st') = evalExp (e, getExpType e "null" st)  [] st
                              in e'

evalClsDecls :: [CL] -> Stack -> Stack
evalClsDecls [] st = st
evalClsDecls (cl:cls) st = evalClsDecls (cls) (evalCl cl st)

evalCl :: CL -> Stack -> Stack
evalCl (Class id1 id2 cons fs ms) st = (id1,id2,evalConst(cons), evalField(fs), evalMethod(ms)):st

evalConst :: Const -> ([(Type,Id)], ConsBody)
evalConst (Cons id fs bd) = (fs, bd)

evalF :: Field -> FIeld
evalF (Fd t id) = (t, id, Null)


evalField :: [Field] -> [FIeld]
evalField []        = []
evalField (f:fs)    = evalF(f):evalField(fs)


evalMethod :: [Method] -> [Methd]
evalMethod []                  = []
evalMethod ((Meth t id fs bd):ms) = (t, id, fs, bd):evalMethod(ms)

getState :: [Id] -> [Exp] -> State
getState [] [] = []
getState (id:ids) (e:es) = (id,e):(getState ids es)

getIdExp :: Id -> State -> Exp
getIdExp id ((id1,e):s)  
            | id == id1 = e
            | otherwise = getIdExp id s

--- expressions evaluation

type State = [(Id,Exp)]

evalExp :: (Exp, Type) -> State -> Stack -> (Exp,Type,State)

evalExp (New t es,t')  s  st = let es' = reduceExp es t' s st
                               in (New t (getFst es'), t, getState (getFieldNameList (fieldLookup t st)) (getFst es'))

evalExp (Get e id,t)   s  st= let (e1, t', s') = evalExp (e, t) s st
                              in let e2 = getIdExp id s'
                                   in let (_, _, s'') = evalExp (e2, t') s st 
                                      in (e2, t', s'') 
evalExp (Cast t2 e,t)  s st = let (e', t', s') = evalExp (e, t) s st
                                       in if (elem t2 (getSuperClassList t' st)) then (e',t2,s') else (Null, "Type error", s')

evalExp (Call ex m es,t') s st =   let (e,t1,s') = evalExp (ex, t') s st
                                   in let (t'',_, fs, bd) =  getMethod m (methodLookup t1  st)
                                              in let es' = reduceExp es t' s st
                                                 in let s'' = (getState1 (getFst es') fs)++s'
                                                        in evalExp (evalMethodBody e bd s'' t1 st, t'') s'' st
evalExp (This id, t)   s   st = (getExp (getElemByID id s), t, s)
evalExp (ID id, t)     s   st = (getExp (getElemByID id s), t, s)

reduceExp :: [Exp] -> Type -> State -> Stack -> ([(Exp,Type)])
reduceExp [] t s st = []
reduceExp (e:es) t s st = let (e', t', s') = evalExp (e,t) s st 
                        in (e',t'):(reduceExp es t s st)

getState1 :: [Exp] -> [(Type, Id)] -> State
getState1 [] [] = []
getState1 (e:es) ((t,id):fs) = (id,e):(getState1 es fs)

--- bodies evaluations

evalMethodBody :: Exp -> MethBody -> State -> Type -> Stack -> Exp
evalMethodBody  e (New id es)    s t st = let es' = reduceExp es t s st
                                          in New id (getFst es') 
evalMethodBody  e (This id)      s t st = let (e',_,_) = evalExp (Get e id,t) s st
                                          in e'
evalMethodBody  e (Call e1 m es) s t st = Call (evalMethodBody e e1 s t st) m (matchEs es s t st) 
evalMethodBody  e (Cast id e1)   s t st = Cast id (evalMethodBody e e1 s t st)
evalMethodBody  e (ID id)        s t st = getExp (getElemByID id s)

matchEs :: [Exp] -> State -> Type -> Stack -> [Exp]
matchEs [] s t st = []
matchEs ((ID id):es)   s t st       = (getExp (getElemByID id s)):(matchEs es s t st)
matchEs ((This id):es) s t st       = (getExp (getElemByID id s)):(matchEs es s t st)
matchEs ((New id es'):es) s t st    = (get (evalExp (New id es', t) s st)):(matchEs es s t st)
matchEs ((Call e1 m es'):es) s t st = let (e,_,_) = evalExp (Call e1 m es',t) s st
                                       in e:(matchEs es s t st)
get :: (Exp, Type, State) -> Exp
get (e,_,_) =e

getElemByID :: Id -> State -> (Id,Exp)
getElemByID id (s:ss)
              | id == getID s =  s
              | otherwise = getElemByID id ss

getID :: (Id, Exp) -> Id
getID (id,_) = id

getExp :: (Id,Exp) -> Exp
getExp (_,e) = e

getExpType :: Exp -> Type -> Stack  -> Type
getExpType (Null)         t  st = "null"
getExpType (New  t es)    t' st = t
getExpType (Get  e id)    t  st = getExpType e  t st
getExpType (Cast t' e2)   t  st = t'
getExpType (Call e id es) t  st = getExpType e  t st
getExpType (ID id)        t  st = getFieldType (getField id (fieldLookup t st))
getExpType (This id)      t  st = getExpType (ID id) t st

getFst :: [(Exp, Type)] -> [Exp]
getFst [] = []
getFst ((e,_):et) = e:(getFst et)

---- interpreter

program2str :: Program -> String
program2str (Prog cls body) = class2str(cls)++" class Main { public static void main (String[] args) { "++ exp2str(body)++ " } }"

class2str :: [CL] -> String
class2str [] = " "
class2str (cl:cls) = cl2str(cl)++class2str(cls)

cl2str :: CL -> String
cl2str (Class id1 id2 const fs ms) = "class "++id1++" extends "++id2++" { "++cons2str(const)++" "++field2str(fs)++" "++method2str(ms)++" }"

cons2str :: Const -> String
cons2str (Cons id fs body) = id++ "("++args2str(fs)++") { "++body2str(body)++" }"

args2str :: [(Type, Id)] -> String
args2str [] = ""
args2str ((t,id):as) = if (as==[]) then t++" "++id else t++" "++id++"; "++args2str(as)

body2str :: ConsBody -> String
body2str (Super s1 s2 s3) = "super( "++ids2str(s1)++" ); "++(init2str s1 s2)

init2str :: [Id] -> [Id] -> String
init2str [] [] =""
init2str s1 s2 = "this."++(head s1)++" = "++(head s2)++"; "++(init2str (tail s1) (tail s2))

field2str :: [Field] -> String
field2str [] = ";"
field2str (f:fs) = if (fs == []) then fld2str(f)++";" else fld2str(f)++"; "++field2str(fs)

fld2str :: Field -> String
fld2str (Fd id1 id2)= id1++" "++id2

method2str :: [Method] -> String
method2str [] =" "
method2str (m:ms) = mt2str(m)++" "++method2str(ms)

mt2str :: Method -> String
mt2str (Meth t f fs body) = t++" "++f++" ("++args2str(fs)++") { "++exp2str(body)++"; }"

ids2str :: [Id] -> String
ids2str [] =""
ids2str (id:ids) = if (ids==[]) then id else id++"; "++ids2str(ids)

exp2str :: Exp -> String
exp2str (ID id)    = id
exp2str (Get e id) = exp2str(e)++"."++id
exp2str (Call e id es) = exp2str(e)++"."++id++"( "++lexp2str(es)++" )"
exp2str (Cast id e) = "("++id++")"++exp2str(e)
exp2str (This id) = "this."++id
exp2str (New id es) = "new "++id++" ("++lexp2str(es)++")"

lexp2str :: [Exp] -> String
lexp2str [] = ""
lexp2str (e:es) = if (es == []) then exp2str(e) else exp2str(e)++"; "++lexp2str(es)

--- Test Programs

prog1 :: Program
prog1 = Prog
        [ (Class "A"    "Object"  (Cons "A" [] (Super [] [] [])) [] []),
          (Class "B"    "Object"  (Cons "B" [] (Super [] [] [])) [] []),
          (Class "Pair" "Object"  (Cons "Pair" [("Object","fst"),("Object","snd")] (Super ["fst","snd"] ["fst","snd"] ["fst","snd"])) 
                                  [(Fd  "Object" "fst"),  (Fd  "Object" "snd")] 
                                  [(Meth "Pair" "setfst" [("Object","newfst")]  (New "Pair" [(ID "newfst"), (This "snd")]))])
        ]
       --(New "Pair" [(New "Pair"[(New "A"[]),(New "B" [])]), (New "Pair" [(New "B" []),(New "B" [])])]) 
       (Call (New "Pair" [(New "A" []),(New "B" [])]) "setfst" [(New "B" [])])

prog3 :: Program
prog3 = Prog
        [ (Class "A"    "Object"  (Cons "A" [] (Super [] [] [])) [] []),
          (Class "B"    "Object"  (Cons "B" [] (Super [] [] [])) [] []),
          (Class "Pair" "Object"  (Cons "Pair" [("Object","fst"),("Object","snd")] (Super ["fst","snd"] ["fst","snd"] ["fst","snd"])) 
                                  [(Fd  "Object" "fst"),  (Fd  "Object" "snd")] 
                                  [(Meth "Pair" "setfst" [("Object","newfst")] (New "Pair" [(ID "newfst"), (This "snd")]))])
        ]      
        (Get (New "Pair" [(New "A" []),(New "B" [])]) "snd")

prog4 :: Program
prog4 = Prog
        [ (Class "A"    "Object"  (Cons "A" [] (Super [] [] [])) [] []),
          (Class "B"    "Object"  (Cons "B" [] (Super [] [] [])) [] []),
          (Class "Pair" "Object"  (Cons "Pair" [("Object","fst"),("Object","snd")] (Super ["fst","snd"] ["fst","snd"] ["fst","snd"])) 
                                  [(Fd  "Object" "fst"),  (Fd  "Object" "snd")] 
                                  [(Meth "Pair" "setfst" [("Object","newfst")] (New "Pair" [(ID "newfst"), (This "snd")]))])
        ] 
        (Get (Cast "Pair" (Get (New "Pair" [(New "Pair" [(New "A" []),(New "B" [])]), (New "A" [])]) "fst")) "snd")

prog2 :: Prg
prog2 = 
        [ (Class "A"    "Object"  (Cons "A" [] (Super [] [] [])) [] []),
          (Class "B"    "Object"  (Cons "B" [] (Super [] [] [])) [] []),
          (Class "Pair" "Object"  (Cons "Pair" [("Object","fst"),("Object","snd")] (Super ["fst","snd"] ["fst","snd"] ["fst","snd"])) 
                                  [(Fd  "Object" "fst"),  (Fd  "Object" "snd")] 
                                  [(Meth "Pair" "setfst" [("Object","newfst")] (New "Pair" [(ID "newfst"), (This "snd")]))])
        ] 
      
prog5 :: Program
prog5 =  Prog
        [ (Class "Entier" "Object" (Cons "Entier" [] (Super [] [] [])) [] [(Meth "Entier" "Add" [("Entier","x")] (ID "x"))]),
          (Class "Zero"   "Entier" (Cons "Zero"   [] (Super [] [] [])) [] [(Meth "Entier" "Add" [("Entier","x")] (ID "x"))]),
          (Class "Succ"   "Entier" (Cons "Succ"   [("Entier","next")] (Super [] ["next"] ["next"])) [(Fd "Entier" "next")] 
                                   [(Meth "Entier" "Add" [("Entier","x")] (Call (This "next") "Add" [(New "Succ" [(ID "x")])]))])
        ]
       --(Call (New "Succ" [(New "Zero" [])]) "Add" [(New "Zero" [])])
       (Call (New "Succ" [(New "Succ" [(New "Succ" [(New "Succ" [(New "Zero"[])])])])]) "Add" [(New "Succ" [(New "Succ" [(New "Succ" [(New "Zero"[])])])])])       

prog6 :: Prg
prog6 = 
        [ (Class "Entier" "Object" (Cons "Entier" [] (Super [] [] [])) [] [(Meth "Entier" "Add" [("Entier","x")] (ID "x"))]),
          (Class "Zero"   "Entier" (Cons "Zero"   [] (Super [] [] [])) [] [(Meth "Entier" "Add" [("Entier","x")] (ID "x"))]),
          (Class "Succ"   "Entier" (Cons "Succ"   [("Entier","next")] (Super [] ["next"] ["next"])) [(Fd "Entier" "next")] 
                                   [(Meth "Entier" "Add" [("Entier","x")] (Call (This "next") "Add" [(New "Succ" [(ID "x")])]))])
        ]


exp :: Exp
exp =  (Call (New "Pair" [(New "A" []),(New "B" [])]) "setfst" [(New "B" [])])

exp1 = (Get (Cast "Pair" (Get (New "Pair" [(New "Pair" [(New "A" []),(New "B" [])]), (New "A" [])]) "fst")) "snd")

exp2 = (Get (New "Pair" [(New "Pair" [(New "A" []),(New "B" [])]), (New "A" [])]) "snd")
