package costo.kml2java.framework;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;

import costo.kml2java.framework.channels.Channel;
import costo.kml2java.framework.channels.IChannel;
import costo.kml2java.framework.config.ExecutionParameters;
import costo.kml2java.framework.exceptions.KmlCommunicationException;
import costo.kml2java.framework.exceptions.ServiceException;
import costo.kml2java.framework.exceptions.UnsatisfiedPostconditionException;

/**
 * Mother of all executable services
 * 
 * subclasses add their own states, public methods corresponding to messages (and subservice calls) package methods for guards and transition
 * evaluation
 * 
 * TODO PA : a part of the pre/post processing is to move at the ServiceAtExecution level
 * 
 * @author ardourel-g TODO PA : restructured the class generalise the common part at the ServiceAtExecution superclass level include the scheduler
 *         work here
 * 
 *         synchronized policy to study in details
 * 
 */
public abstract class ProvidedService extends Service implements Runnable, IProvidedService {
	private int sleepTime = ExecutionParameters.pause;
	private boolean paused = ExecutionParameters.paused;

	private boolean blockingCalls = true;
	private ServiceState runningState = ServiceState.CREATED;
	private LockCase lock = LockCase.unlocked;
	private IPauseManager pauseManager;
	private IServiceExecutionPolicy executionPolicy = new DefaultServiceExecutionPolicy();
	protected ExecutableLTS lts;
	protected Thread serverT;

	private Map<String, IRequiredService> required = new HashMap<String, IRequiredService>();
	private Map<String, Channel> intChannels = new HashMap<String, Channel>();

	Random r = new Random(); // copied from sched

	// ////////////// Service Access ////////////////

	/**
	 * @return the debug
	 */
	public boolean isDebug() {
		return debug;
	}

	/**
	 * @param debug
	 *            the debug to set
	 */
	synchronized public void setDebug(boolean debug) {
		this.debug = debug;
	}

	// @Override
	/**
	 * @see costo.kml2java.framework.IProvidedService#isProvided()
	 */
	@Override
	public Boolean isProvided() {
		return true;
	}

	/**
	 * @return the lts
	 */
	@Override
	public ExecutableLTS getLTS() {
		return lts;
	}

	/**
	 * @param lts
	 *            the lts to set
	 */
	synchronized public void setLTS(ExecutableLTS lts) {
		this.lts = lts;
	}

	/**
	 * @see superclass (base channel) = client channel
	 * @see required services = provider's channel
	 */
	// @Deprecated
	// public void initChannels() {
	// }

	/**
	 * @see costo.kml2java.framework.IProvidedService#setCallChannel(costo.kml2java.framework.channels.Channel)
	 */
	@Override
	synchronized public void setCallChannel(Channel ch) {
		this.assignChannel(ch);
	}

	private void closeCallChannel() {
		this.assignChannel(null);
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#getRequiredFromChannelName(java.lang.String)
	 */

	public IRequiredService getRequiredFromChannelName(String ch) {
		return required.get(ch.substring(1, ch.length()));
		// removes the "_"
	}

	/*
	 * @Deprecated public IChannel getChannel(String ch) { IChannel chan = this.getChannels().get(ch); // System.out.println(this.name +
	 * this.hashCode() + " " + ch + "is " + chan + " in " + this.channels);
	 * 
	 * if (chan == null) // FIXME : should raise an error instead return RandomChannel.instance; return chan; }
	 */

	@Override
	public void setResult(Object object) {

	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#isAlive()
	 */
	@Override
	public boolean isAlive() {
		return runningState == ServiceState.ALIVE;
	}

	public boolean isStopped() {
		return runningState == ServiceState.STOPPED;
	}

	public boolean isReady() {
		return runningState == ServiceState.READY;
	}

	public boolean isFinished() {
		return runningState == ServiceState.FINISHED;
	}

	private boolean isEnding() {
		return runningState == ServiceState.ENDING;
	}

	public String printStatus() {
		return runningState.toString();
	}

	/**
	 * @return the locked
	 */
	@Override
	public boolean isLocked() {
		return (lock != LockCase.unlocked);
	}

	/**
	 * @return the locked
	 */
	@Override
	public boolean isLockedOn(LockCase reason) {
		return (lock == reason);
	}

	/**
	 * @param locked
	 *            lock the service (with a reason argument)
	 */

	public void lock(LockCase reason) {
		this.lock = reason;
	}

	/**
	 * unlock the service
	 */

	public void unlock() {
		this.lock = LockCase.unlocked;
	}

	// //////////////Service Thread Access ////////////////

	/**
	 * @see costo.kml2java.framework.IProvidedService#start()
	 */
	@Override
	public void start() throws ServiceException {
		if (this.isAlive() || this.isStopped())
			return;
		if (lts == null)
			throw new ServiceException(this, "has null lts");
		if (this.isFinished()) {
			this.init();
		}
		this.serverT = new Thread(this);
		this.unlock();
		this.runningState = ServiceState.ALIVE;
		// LOG OFF System.out.println("starting service thread" + serverT);
		serverT.start(); // run in the thread
	}

	/*
	 * public void suspend() { serverT.suspend(); }
	 * 
	 * public void resume() { serverT.resume(); }
	 */

	// ////////////// Service Running ////////////////

	/**
	 * @see costo.kml2java.framework.IProvidedService#reset()
	 */
	@Override
	synchronized public void reset() {
		this.lts.reset();
		this.initState();
		this.fireServiceEvent("", "reset", null);
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#init()
	 */
	@Override
	public void init() {
		initState();
		initLTS();
		this.runningState = ServiceState.READY;
	}

	synchronized public void initState() {
		// must be redefined in the subclasses
	}

	public void initLTS() {
		// FIXME link to LTS in the translation ?
		lts.init();
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#executeStartingTransaction()
	 */
	@Override
	public void executeStartingTransaction() {
		// will be generated in the subclasses for each application
		// example :
		// this.receiveServiceCall("__CALLER","Manager_newReference",new
		// Class<?>[]{},this);
	}

	// FIXME add thread commands or indirections ?
	/**
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		while (this.isPaused()) {
			Thread.yield();
		}
		lts.start(); // launch the service execution
		while (this.isAlive()) {

			if (!this.isLocked() && !this.isPaused()) {
				if (lts.finalStateReached()) {
					runningState = ServiceState.ENDING;
				} else {
					nextStep();
					if (pauseManager != null) {
						this.paused = pauseManager.shouldPause(this);
					}
				}
			}
			// Thread.yield();
			if (!this.isFinished()) {
				try {
					Thread.sleep(sleepTime);
				} catch (InterruptedException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		if (this.debug) System.out.println(">> service " + this.getLongName() + " is finished in state "+runningState.toString());
		this.closeRequirements();
		runningState = ServiceState.FINISHED;
		if (lts.terminalStateReached()) {
			if (this.debug) System.out.println(this.getLongName() + "service terminal");
			this.getOwner().stopSystem();
		}
		// LOG OFF System.out.println("End of Service Run" + this.serverT);
	}

	/**
	 * executes next step of the LTS. Delegates to its ServiceExecutionPolicy
	 * 
	 * @return true if another step can be taken
	 */
	private boolean nextStep() {
		// FIXME what to do ?
		this.executionPolicy.nextStep();
		// if (!this.alive)
		// return false;
		if (lts != null) {
			// if (lts.finalStateReached())
			// return false;
			ArrayList<String> possible = lts.getPossibleOutGoingTransitions();

			if (possible == null || possible.size() == 0) {
				if (debug) {
					System.err.println("no more transitions from " + lts.getService());
				}
				return false;
			}

			String selected = selectTransitionFrom(possible);
			if (debug) {
				System.out.println(selected + " " + lts.getService() + " " + lts.getService().getOwner());
				// TODO : properly log trace
			}
			try {
				lts.executeTransition(selected);
			} catch (Exception e) {
				// TODO : properly log trace
				System.err.println("Exception occured when executing transition " + selected);
				e.printStackTrace();
			}
			return true;
		}
		return false;
	}

	// public String removeUnReachable
	/**
	 * @param possible
	 * @return FIXME useful ?
	 */
	public String selectTransitionFrom(ArrayList<String> possible) {
		return possible.get(r.nextInt(possible.size()));
	}

	/**
	 * Normal end of service
	 */
	private void closeRequirements() {
		if (! this.isEnding())
			System.err.println("Error: closing requirements of " + this.getLongName()+" while not finished");
		else {
			this.fireServiceEvent("closeRequirements", "end of service", null);
			this.stopIntRequired();
			this.stopRequired();
		}	
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#stop() owner interruption
	 */
	@Override
	public void stop() {
		stop("no reason given");
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#stop(java.lang.String) owner interruption
	 */
	public void stop(String reason) {
		if (this.isStopped())
			return;
		if (this.isAlive()) {
			runningState = ServiceState.STOPPED;
		}
		this.fireServiceEvent("stopped", reason, null);
		this.stopIntRequired();
		this.stopRequired();
	}

	private void stopIntRequired() {
		for (Channel ic : this.intChannels.values()) {
			if (ic != null) {
				ic.cut(this);
			}
		}
	}

	private void stopRequired() {
		for (IRequiredService r : this.required.values()) {
			if (r != null) {
				r.stop();
				// FIXME what about the promoted ?
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see costo.kml2java.framework.IService#stop(costo.kml2java.framework.channels .IChannel) client interruption
	 */
	public void stop(IChannel source) {
		this.stop("interrupted client");
	}

	/**
	 * @param source
	 *            end of communication : the channel is normally closed
	 */
	public void eoc(IChannel source) {
		this.fireServiceEvent("", source.getName() + "reset", null);
		this.closeCallChannel();
	}

	// //////////////Service Communication ////////////////

	/*
	 * (non-Javadoc)
	 * 
	 * @see costo.kml2java.framework.ServiceAtExecution#fireCommunicationEvent(java .lang.String, java.lang.String, java.lang.Exception)
	 */
	/**
	 * @see costo.kml2java.framework.IProvidedService#fireServiceEvent(java.lang.String, java.lang.String, java.lang.Exception)
	 */
	@Override
	public void fireServiceEvent(String tname, String log, Exception ex) {
		for (IServiceEvolutionListener l : this.listeners) {
			l.transitionFired(tname, (this.lts != null ? lts.getCurrentState() : ""), this, log, ex);
		}
	}

	private IRequiredService getRequiredServiceForChannel(String channame) {
		String servname = channame.substring(1);
		IRequiredService req = this.required.get(servname);
		if (req == null) {
			req = this.getOwner().getRequiredService(servname);
			this.required.put(servname, req);
		}
		return req;
	}

	private Channel getIntRequiredChannel(String channame) {
		// removes the "_" prefix
		String servname = channame.substring(1);
		// System.out.println("channel for name "+servname);
		Channel chan = this.intChannels.get(servname);
		// suppose servname is unique for the component
		if (chan == null) {
			IProvidedService serv = this.getOwner().callService(servname);
			chan = this.getOwner().getIntChannel(servname, this);
			this.intChannels.put(servname, chan);
			serv.setCallChannel(chan);
			System.out.println(chan.getLongName() + "-> new channel for service " + serv.getName());
		}
		// System.out.println("channel for "+servname+" " +chan.toString());
		return chan;
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#receiveMessage(java.lang.String, java.lang.String, java.lang.Class,
	 *      costo.kml2java.framework.Service)
	 */
	@Override
	public Object[] receiveMessage(String channel, String message, Class<?>[] params, IService orig) {
		if (isIntreq(channel))
			return this.getIntRequiredChannel(channel).receiveMessage(channel, message, params, orig);
		else if (channel.equals("__CALLER"))
			return this.getChannel().receiveMessage(channel, message, params, orig);
		else
			return this.getRequiredFromChannelName(channel).receiveMessage(channel, message, params, orig);
		// FIXME : should come from to the required or the caller
		// IChannel chan = this.getChannel(channel);
		// if (chan != null)
		// return chan.receiveMessage(channel, message, params, orig);
		// return null;
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#emitMessage(java.lang.String, java.lang.String, java.lang.Object[],
	 *      costo.kml2java.framework.Service)
	 */
	@Override
	public void emitMessage(String channel, String message, Object[] params, IService orig) {
		if (this.blockingCalls) {
			this.lock(LockCase.lockedOnSend);
		}
		if (isIntreq(channel)) {
			this.getIntRequiredChannel(channel).emitMessage(channel, message, params, orig);
		} else if (channel.equals("__CALLER")) {
			this.getChannel().emitMessage(channel, message, params, orig);
		} else {
			this.getRequiredFromChannelName(channel).emitMessage(channel, message, params, orig);
		}

	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#callService(java.lang.String, java.lang.String, java.lang.Object[],
	 *      costo.kml2java.framework.Service)
	 */
	@Override
	public void callService(String channel, String message, Object[] params, IService orig) {
		if (this.blockingCalls) {
			this.lock(LockCase.lockedOnCall);
		}

		if (isIntreq(channel)) {
			try {
				this.getIntRequiredChannel(channel).callService(channel, message, params, orig);
				// FIXME we change the caller
			} catch (KmlCommunicationException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
			// FIXME call on self
		} else if (channel.equals("__CALLER")) {
			// FIXME call sub on CALLER
		} else {
			this.getRequiredServiceForChannel(channel).callService(channel, message, params, orig);
		}

	}

	private boolean isIntreq(String string) {

		String chstring = string.substring(1);
		if (this.getOwner().getService(chstring) != null)
			return true;
		return false;
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#returnService(java.lang.String, java.lang.String, java.lang.Object[],
	 *      costo.kml2java.framework.Service)
	 */
	@Override
	public void returnService(String channel, String message, Object[] params, IService orig) {
		this.lock(LockCase.lockedOnReturn);
		if (params.length > 0) {
			this.setResult(params[0]);
		}
		if (!this.isPostSatisfiedWithTheseParameters(params)) {

			this.fireServiceEvent("err", this.getLongName() + "post failed", new UnsatisfiedPostconditionException(this, this.getName()
					+ "post failed"));

		}
		Channel chan = this.getChannel();
		if (chan != null) {

			chan.returnService(channel, message, params, orig);

			// if (channel.equals("__CALLER")) {
			// FIXME : should always be true
			// System.err.println("__caller taken in list");
			// this.getChannel().put("__CALLER", this.callers_waiting.poll());
			// }
		} else {
			System.err.println(" no channel found in " + this);
			// FIXME throw new ChannelException(this, "no channel", chan);
		}
	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#receiveServiceCall(java.lang.String, java.lang.String, java.lang.Class,
	 *      costo.kml2java.framework.Service)
	 */

	@Override
	public Object[] receiveServiceCall(String channel, String message, Class<?>[] paramtypes, IService orig) {

		return this.getChannel().receiveServiceCall(channel, message, paramtypes, orig);

	}

	/**
	 * @see costo.kml2java.framework.IProvidedService#receiveServiceReturn(java.lang.String, java.lang.String, java.lang.Class,
	 *      costo.kml2java.framework.Service)
	 */
	@Override
	public Object[] receiveServiceReturn(String channel, String message, Class<?>[] paramtypes, IService orig) {

		if (isIntreq(channel))
			return this.getIntRequiredChannel(channel).receiveServiceReturn(channel, message, paramtypes, orig);
		else {
			IRequiredService rs = this.getRequiredFromChannelName(channel);
			if (rs != null)
				return rs.receiveServiceReturn(channel, message, paramtypes, orig);
			return null;
		}
	}

	// ////////////// Service Assertions ////////////////

	public Boolean isVirtualInvSatisfied() {

		return true;
	}

	/**
	 * @return the usingPreconditionsAsGuards
	 */
	public boolean isUsingPreconditionsAsGuards() {
		return usingPreconditionsAsGuards;
	}

	@Override
	public void ack(Channel chan, int commtype) {
		// System.out.println("--ACK " + chan.toString() + commtype);
		if (chan == this.baseChannel && isLockedOn(LockCase.lockedOnReturn)) {
			// chan.close(this); client responsability
		}
		// // the required services will unlock myself in the other cases
		// if (this.isLockedOn(LockCase.lockedOnSend)) {
		// this.unlock();
		// } else
		// throw new ServiceException(this, "lock error");
		// } else
		// throw new ChannelException(this, "not the good channel", chan);
		this.unlock();
	}

	/**
	 * @see costo.kml2java.framework.IService#getStateSpace()
	 */
	@Override
	public String getStateSpace() {
		return "   lock(" + this.lock + ") state(" + this.runningState + ")";
	}

	/**
	 * @return the paused
	 */
	public boolean isPaused() {
		return paused;
	}

	/**
	 * @param paused
	 *            the paused to set
	 */
	public void setPaused(boolean paused) {
		this.paused = paused;
	}

	/**
	 * @return the sleepTime
	 */
	public int getSleepTime() {
		return sleepTime;
	}

	/**
	 * @param sleepTime
	 *            the sleepTime to set
	 */
	public void setSleepTime(int sleepTime) {
		this.sleepTime = sleepTime;
	}

	/**
	 * @return the blockingCalls
	 */
	public boolean isBlockingCalls() {
		return blockingCalls;
	}

	/**
	 * @param blockingCalls
	 *            the blockingCalls to set
	 */
	public void setBlockingCalls(boolean blockingCalls) {
		this.blockingCalls = blockingCalls;
	}

	/**
	 * @return the pauseManager
	 */
	public IPauseManager getPauseManager() {
		return pauseManager;
	}

	/**
	 * @param pauseManager
	 *            the pauseManager to set
	 */
	@Override
	public void setPauseManager(IPauseManager pauseManager) {
		this.pauseManager = pauseManager;
	}

}
