/**
 * An ExecutionContext represents the location where
 * components or composites instance evolve.
 * This includes:
 *   - the component management component 
 *      - instance creation and initialisation
 *      - service call
 *   - the component and middleware management
 *      - channel management
 *      - communication management
 *      - scheduling and allocation policy ? 
 */
package costo.kml2java.framework;

import java.util.ArrayList;
import java.util.HashMap;

import costo.kml2java.framework.channels.Channel;
import costo.kml2java.framework.channels.DefaultChannelListener;
import costo.kml2java.framework.channels.ExecutionBinding;
import costo.kml2java.framework.channels.IChannelListener;

/**
 * @author Pascal
 * 
 */
public abstract class ExecutionContext {

	/**
	 * just to log information
	 */
	protected String name;

	private ExecutableComponent owner = null;

	/**
	 * instances are component instances names are supposed to be unique, those of the declaration c : C the type C is given by the
	 * ExecutableComponent
	 */
	protected HashMap<String, ExecutableComponent> instances;

	/**
	 * key: cid.clientservicename bindings are the static description of (assembly) links service call will instantiate channels from bindings on the
	 * fly (demand)
	 */
	protected HashMap<String, ExecutionBinding> bindings;

	protected ArrayList<IChannelListener> channelsListeners = new ArrayList<IChannelListener>();

	/*
	 * TODO choix � r�gler
	 * 
	 * cr�ation en une fois � partir des liens Kmelia .
	 * 
	 * 2 passes 1. noms 2. instances
	 */

	public ExecutionContext(String name) {
		super();
		this.name = name;
		this.instances = new HashMap<String, ExecutableComponent>();
		this.bindings = new HashMap<String, ExecutionBinding>();
		this.addChannelListener(new DefaultChannelListener());
	}

	// lazy fulfillment

	public ExecutionContext(String name, HashMap<String, ExecutableComponent> instances, HashMap<String, ExecutionBinding> bindings) {
		this(name);
		this.instances = instances;
		this.bindings = bindings;
		for (ExecutionBinding b : this.bindings.values()) {
			this.addListeners(b);
		}
	}

	private void addListeners(ExecutionBinding b) {
		for (IChannelListener l : this.channelsListeners) {
			b.addChannelListener(l);
		}
	}

	// how do we create the bindings ? by generation ?

	/**
	 * @param cname
	 *            component name
	 * @param c
	 *            component instance
	 * @return true if the instance is new overriden is allowed
	 */
	public boolean addInstance(String cname, ExecutableComponent c) {
		ExecutableComponent cbefore = instances.put(cname, c);
		return (cbefore == null);
	}

	// remove ?

	/**
	 * @param bname
	 *            binding name cid.clientservicename
	 * @param b
	 *            binding
	 * @return true if the binding is new overriden is allowed
	 */
	public boolean addBinding(ExecutionBinding b) {
		ExecutionBinding bbefore = bindings.put(b.getCclientname() + b.getSclientname(), b);
		this.addListeners(b);
		return (bbefore == null);
	}

	// remove ?

	public String getName() {
		return name;
	}

	public HashMap<String, ExecutableComponent> getInstances() {
		return instances;
	}

	public ExecutableComponent getComponentFromName(String cname) {
		if (cname.equals("SELF"))
			return this.getOwner();
		return instances.get(cname);
	}

	protected HashMap<String, ExecutionBinding> getBindings() {
		return bindings;
	}

	public ExecutionBinding getBindingFromName(String bname) {
		return bindings.get(bname);
	}

	public void addChannelListener(IChannelListener l) {
		this.channelsListeners.add(l);
	}

	/**
	 * entry point main static method ?
	 */
	public void start() {
		//System.out.print("System initialisation");
	}

	/**
	 * @param clientComponent
	 * @param clientService
	 * @return the code will depend from the kind of service
	 */
	public Channel instantiateLink(ExecutableComponent clientComponent, IRequiredService clientService) {
		// FIXME
		ExecutionBinding binding = this.getBindingFromName(clientComponent.getIdJFK() + clientService.getName());
		// get an AssemblyBinding or a RPromotionBinding
		return binding.instantiateLink("chantype", clientComponent, clientService);
	}

	public void closeChannel(ExecutableComponent executableComponent, IService clientservice, Channel chan) {
		// chan.getOwner().closeChannel(this, clientservice, chan);
		// FIXME what about promoted required ??
		chan.close(clientservice);
	}
	
	public void cutChannel(ExecutableComponent executableComponent,
			IService clientservice, Channel chan) {
		chan.cut(clientservice);
	}

	/**
	 * @return the owner
	 */
	public ExecutableComponent getOwner() {
		return owner;
	}

	/**
	 * @param owner
	 *            the owner to set
	 */
	public void setOwner(ExecutableComponent owner) {
		this.owner = owner;
	}

}
