package toolbox.editors;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.presentation.IPresentationReconciler;
import org.eclipse.jface.text.presentation.PresentationReconciler;
import org.eclipse.jface.text.rules.BufferedRuleBasedScanner;
import org.eclipse.jface.text.rules.DefaultDamagerRepairer;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewerConfiguration;

import toolbox.Activator;

/**
 * La classe SourceViewerConfig configure l'affichage du texte (indentation, coloration syntaxique, completion automatique).
 */
public class SourceViewerConfig extends SourceViewerConfiguration {
	/** 
	 * Classe utilisee lorsque l'on ne désire pas affiner la coloration syntaxique
	 * (ie. toute la partition de la même couleur). Utilisee pour les partitions
	 * de commentaires.
	 */
	static class SingleTokenScanner extends BufferedRuleBasedScanner {
		/**
		 * Constructeur. Definit le token renvoye.
		 * @param attribute Attribut du Token.
		 */
		public SingleTokenScanner(TextAttribute attribute) {
			setDefaultReturnToken(new Token(attribute));
		}
	}
	
	/**
	 * Renvoie l'identifiant du partitionnement utilise.
	 * @param sourceViewer
	 * @return L'identifiant du type du partitionnement.
	 */
	public String getConfiguredDocumentPartitioning(ISourceViewer sourceViewer) {
		return Activator.TOOLBOX_PARTITIONING;
	}
	
	/**
	 * Renvoie les types de partitions qu'il est possible de rencontrer dans le document.
	 * @param sourceViewer SourceViewer pour lequel on renvoie les partitions.
	 * @return Un tableau contenant les identifiants de tous les types de partitions possibles.
	 */
	public String[] getConfiguredContentTypes(ISourceViewer sourceViewer) {
		// Commentaires et lignes classiques
		return new String[] {ToolboxPartitionScanner.TOOLBOX_COMMENT, IDocument.DEFAULT_CONTENT_TYPE};
	}
	
	/**
	 * Met en place la coloration syntaxique.
	 * Attribue des damagers et des repairers à chaque type de partition.
	 * @param sourceViewer SourceViewer pour lequel on configure le reconciler.
	 * @return Le reconciler à utiliser avec les damager et les repairers configurés.
	 */
	public IPresentationReconciler getPresentationReconciler(ISourceViewer sourceViewer) {
		ToolboxTextAttributeProvider provider= Activator.getDefault().getTextAttributeProvider();
		PresentationReconciler reconciler= new PresentationReconciler();
		reconciler.setDocumentPartitioning(getConfiguredDocumentPartitioning(sourceViewer));
		
		// Cree le damager/repairer pour le code
		DefaultDamagerRepairer dr = new DefaultDamagerRepairer(Activator.getDefault().getToolboxCodeScanner());
		reconciler.setDamager(dr, IDocument.DEFAULT_CONTENT_TYPE);
		reconciler.setRepairer(dr, IDocument.DEFAULT_CONTENT_TYPE);

		// Cree le damager/repairer pour les commentaires
		dr= new DefaultDamagerRepairer(new SingleTokenScanner(provider.getAttribute(ToolboxTextAttributeProvider.COMMENT_ATTRIBUTE)));
		reconciler.setDamager(dr, ToolboxPartitionScanner.TOOLBOX_COMMENT);
		reconciler.setRepairer(dr, ToolboxPartitionScanner.TOOLBOX_COMMENT);

		return reconciler;

	}
	
	
	
	
}
