/*
 * @(#)BasicUtilities.java
 *
 * Copyright (c) 2003 Computer Science Research Laboratory,
 * "Babes-Bolyai" University, Cluj-Napoca, ROMANIA.
 * All Rights Reserved.
 */
package ro.ubbcluj.lci.codegen.framework.ocl;

import java.util.List;
import java.util.StringTokenizer;

/**
 * Contains utility methods. Also implements all primitive-type operations specified in OCL
 *
 * @author Cristian Botiza
 */
public final class BasicUtilities {
    
    /**
	 * Retrieves the last token (component) of a string, using the '.' character as token separator
	 *
	 * @param s A string
	 * @return The last component of a string, using '.' as a separator, or null if the supplied string is null
	 */
    static  String getLastComponent(String s) {
		String last = null;
		if (s != null) {
			StringTokenizer st = new StringTokenizer(s, ".", false);
			
			while (st.hasMoreTokens()) {
				last = st.nextToken();
			}
		}
		return last;
    }
    
    /**
	 * Verifies if the two given objects are equal. Also considers the case of null refs (Undefined values)
	 *
	 * @param o1 Any object; may also be null
	 * @param o2 Any object; may also be null
	 * @return true if (o1==null && o2==null) || o1.equals(o2)
	 */
    public static boolean equal(Object o1, Object o2) {
		boolean r;
		if (o1 == null) {
			r = o2 == null;
		}
		else {
			r = o1.equals(o2);
		}
		return r;
    }
    
    /**
	 * The inverse of the equal operation
	 *
	 * @param o1 Any object. May also be null (undefined)
	 * @param o2 Any object. May also be null (undefined)
	 * @return True if (o1 == null && o2 != null) or !o1.equals(o2)
	 */
    public static boolean notEqual(Object o1, Object o2) {
		boolean r;
		if (o1 == null) {
			r = o2 != null;
		}
		else {
			r = !o1.equals(o2);
		}
		return r;
    }
    
    //------------------------------Boolean operations------------------------
    
	/**
	 * The 'isTrue' operation. Returns the supplied value
	 *
	 * @param self A primitive boolean
	 * @return The value of <code>self</code>
	 */
    public static boolean isTrue(boolean self) {
		return self;
    }
	
	/**
	 * The 'isFalse' operation. Returns the negation of the supplied value
	 *
	 * @param self A primitive boolean
	 * @return The value <code>!self</code>
	 */
	
    public static boolean isFalse(boolean self) {
		return !self;
    }
    
    /**
	 * The 'and' operation for boolean values
	 *
	 * @param self A boolean value (on which the 'and' operation is called)
	 * @param arg The argument of the 'and' operation
	 * @return The value <code>self and arg</code>
	 */
    public static boolean and(boolean self, boolean arg) {
		return self && arg;
    }
    
    /**
	 * The 'or' operation for boolean values
	 *
	 * @param self A boolean value (on which the 'or' operation is called)
	 * @param arg The argument of the 'or' operation
	 * @return The value <code>self or arg</code>
	 */
    public static boolean or(boolean self, boolean arg) {
		return self || arg;
    }
    
    /**
	 * The 'xor' operation for boolean values
	 *
	 * @param self A boolean value (on which the 'xor' operation is called)
	 * @param arg The argument of the 'xor' operation
	 * @return The value <code>self xor arg</code>
	 */
    public static boolean xor(boolean self, boolean arg) {
		return self ^ arg;
    }
    
    /**
	 * The 'not' operation for boolean values
	 *
	 * @param self A boolean value (on which the 'not' operation is called)
	 * @return The value <code>not self</code>
	 */
    public static boolean not(boolean self) {
		return !self;
    }
    
    /**
	 * The 'implies' operation for boolean values
	 *
	 * @param self A boolean value (on which the 'implies' operation is called)
	 * @param arg The argument of the 'implies' operation
	 * @return The value <code>self implies arg</code>
	 */
    public static boolean implies(boolean self, boolean arg) {
		return !self || arg;
    }
    
    
    //------------------------------Integer and Real operations-------------
    //These types are equated with the corresponding primitive types provided by Java (int, float)
    
    /**
	 * The 'min' operation for real numbers. This case also includes the comparison
	 * of an integer with a real, in which case a real is returned.
	 *
	 * @param self A real value, on which the 'min' operation is called
	 * @param arg The argument of the 'min' operation
	 * @return The value <code>self.min(arg)</code>
	 */
    public static float min(float self, float arg) {
		return Math.min(self, arg);
    }
    
    
    /**
	 * The 'max' operation for real numbers. This case also includes the comparison
	 * of an integer with a real, in which case a real is returned.
	 *
	 * @param self A real value, on which the 'max' operation is called
	 * @param arg The argument of the 'max' operation
	 * @return The value <code>self.max(arg)</code>
	 */
    public static float max(float self, float arg) {
		return Math.max(self, arg);
    }
    
    /**
	 * The 'divide' operation for real numbers.
	 *
	 * @param self A real number, on which the 'divide' operation is called
	 * @param arg The argument of the 'divide' operation
	 * @return the value <code>self / arg</code>.
	 * If <code>arg</code> is 0.0, Infinity is returned with the corresponding sign.
	 */
    public static float divide(float self, float arg) {
		float r;
		if (arg == 0.0) {
			r = self < 0 ? Float.NEGATIVE_INFINITY : Float.POSITIVE_INFINITY;
		}
		else {
			r = self / arg;
		}
		return r;
    }
    
    /**
	 * The '+' operation for real numbers.
	 *
	 * @param self A real number, on which the '+' operation is called
	 * @param arg The argument of the '+' operation
	 * @return The value <code>self + arg</code>
	 */
    public static float add(float self, float arg) {
		return self + arg;
    }
    
    /**
	 * The '+' operation for integer numbers.
	 *
	 * @param self An integer, on which the '+' operation is called
	 * @param arg The argument of the '+' operation
	 * @return The value <code>self + arg</code>
	 */
    public static int add(int self, int arg) {
		return self + arg;
    }
    
    /**
	 * The '-' operation for real numbers.
	 *
	 * @param self A real number, on which the '-' operation is called
	 * @param arg The argument of the '-' operation
	 * @return The value <code>self - arg</code>
	 */
    public static float subs(float self, float arg) {
		return self - arg;
    }
    
    /**
	 * The '-' operation for integer numbers.
	 *
	 * @param self An integer, on which the '-' operation is called
	 * @param arg The argument of the '-' operation
	 * @return The value <code>self - arg</code>
	 */
    public static int subs(int self, int arg) {
		return self - arg;
    }
    
    /**
	 * The '*' operation for real numbers.
	 *
	 * @param self A real number, on which the '*' operation is called
	 * @param arg The argument of the '*' operation
	 * @return The value <code>self * arg</code>
	 */
    public static float multiply(float self, float arg) {
		return self * arg;
    }
    
    /**
	 * The '*' operation for integer numbers.
	 *
	 * @param self An integer, on which the '*' operation is called
	 * @param arg The argument of the '*' operation
	 * @return The value <code>self * arg</code>
	 */
    public static int multiply(int self, int arg) {
		return self * arg;
    }
    
    /**
	 * The '-' operation for real numbers.
	 *
	 * @param self A real number, on which the '-' operation is called
	 * @return The value <code>-self</code>
	 */
    public static float negate(float self) {
		return -self;
    }
    
    /**
	 * The '-' operation for integer numbers
	 *
	 * @param self An integer, on which the unary '-' operation is called
	 * @return The value <code>-self</code>
	 */
    public static int negate(int self) {
		return -self;
    }
    
    /**
	 * The 'round' operation for real numbers
	 *
	 * @param self A real number, on which the 'round' operation is called
	 * @return The value <code>self.round()</code>
	 */
    public static int round(float self) {
		return Math.round(self);
    }
    
    /**
	 * The 'floor' operation for real numbers
	 *
	 * @param self A real number, on which the 'floor' operation is called
	 * @return The value <code>self.floor()</code>
	 */
    public static float floor(float self) {
		return (float)Math.floor(self);//possible problem here, due to loss of precision (double -> float)
    }
    
    /**
	 * The 'abs' operation for real numbers
	 *
	 * @param self A real number, on which the 'abs' operation is called
	 * @return The value <code>self.abs()</code> (absolute value)
	 */
    public static float abs(float self) {
		return Math.abs(self);
    }
    
    /**
	 * The 'abs' operation for integer numbers
	 *
	 * @param self An integer, on which the 'abs' operation is called
	 * @return The value <code>self.abs()</code>
	 */
    public static int abs(int self) {
		return Math.abs(self);
    }
    
    //---------Integer specific operations---------
    
	/**
	 * The 'max' operation for integer numbers
	 *
	 * @param self An integer, on which the 'max' operation is called
	 * @param arg The argument of the 'max' operation
	 * @return The value <code>self.max(arg)</code>
	 */
    public static int max(int self, int arg) {
		return Math.max(self, arg);
    }
	
	/**
	 * The 'min' operation for integer numbers
	 *
	 * @param self An integer, on which the 'min' operation is called
	 * @param arg The argument of the 'min' operation
	 * @return The value <code>self.min(arg)</code>
	 */
    public static int min(int self, int arg) {
		return Math.min(self, arg);
    }
    
	/**
	 * The 'div' operation for integer numbers (integer division)
	 *
	 * @param self An integer, on which the 'div' operation is called
	 * @param arg The argument of the 'div' operation
	 * @return The value <code>self.div(arg)</code>
	 */
    public static int div(int self, int arg) {
		return self / arg;
    }
    
	/**
	 * The 'mod' operation for integer numbers (modulus operation)
	 *
	 * @param self An integer, on which the 'mod' operation is called
	 * @param arg The argument of the 'mod' operation
	 * @return The value <code>self.mod(arg)</code>
	 */
    public static int mod(int self, int arg) {
		return self % arg;
    }
    
    //--------------------------comparison operations for numbers---------------
    
	/**
	 * The 'lt' operation for real numbers (&lt;)
	 *
	 * @param self A real number, on which the 'lt' operation is called
	 * @param arg The argument of the 'lt' operation
	 * @return The value <code>self.lt(arg)</code>
	 */
    public static boolean lt(float self, float arg) {
		return self < arg;
    }
    
	/**
	 * The 'le' operation for real numbers (&lt;=)
	 *
	 * @param self A real number, on which the 'le' operation is called
	 * @param arg The argument of the 'le' operation
	 * @return The value <code>self.le(arg)</code>
	 */
    public static boolean le(float self, float arg) {
		return self <= arg;
    }
	
	/**
	 * The 'gt' operation for real numbers (&gt;)
	 *
	 * @param self A real number, on which the 'gt' operation is called
	 * @param arg The argument of the 'gt' operation
	 * @return The value <code>self.gt(arg)</code>
	 */
    public static boolean gt(float self, float arg) {
		return self > arg;
    }
	
	/**
	 * The 'ge' operation for real numbers (&gt;=)
	 *
	 * @param self A real number, on which the 'ge' operation is called
	 * @param arg The argument of the 'ge' operation
	 * @return The value <code>self.ge(arg)</code>
	 */
    public static boolean ge(float self, float arg) {
		return self >= arg;
    }
    
    //----------------------------String operations----------------------------
    //The String type in OCL is equated with the java.lang.String type
    
    /**
	 * The 'size' operation for strings
	 *
	 * @param self A String, on which the 'size' operation is called
	 * @return The size (length) of the supplied string. If the string is undefined, undefined is returned
	 * (in this case Integer.MAX_VALUE)
	 */
    public static int size(String self) {
		return self == null ? Integer.MAX_VALUE : self.length();
    }
    
    /**
	 * The 'toLower' operation for strings
	 *
	 * @param self A String, on which the 'toLower' operation is called
	 * @return The lowercased form of the supplied string, or undefined if the string is undefined
	 */
    public static String toLower(String self) {
		String result = null;
		if (self != null) {
			result = new String(self).toLowerCase();
		}
		return result;
    }
    
    /**
	 * The 'toUpper' operation for strings
	 *
	 * @param self A String, on which the 'toUpper' operation is called
	 * @return The uppercased form of the supplied string, or undefined if the string is undefined
	 */
    public static String toUpper(String self) {
		String result = null;
		if (self != null) {
			result = new String(self).toUpperCase();
		}
		return result;
    }
    
    /**
	 * The 'substring' operation for strings
	 *
	 * @param self The string object on which the call is made. May also be null (undefined).
	 * @param start 1-based substring start index
	 * @param stop 1-based substring end index
	 * @return The substring starting at start and ending at stop (inclusive), considering indexes 1-based.
	 * If <code>self</code> is undefined (null), null is returned.
	 */
    public static String substring(String self, int start, int stop) {
		return self == null ? null : self.substring(start-1, stop);
    }
    
    /**
	 * The 'toInteger' operation for strings
	 *
	 * @param self A String, on which the 'toInteger' operation is called
	 * @return The integer value of the given string if it represents an integer number;
	 * otherwise Integer.MAX_VALUE is returned (as an equivalent for Undefined integer values)
	 */
    public static int toInteger(String self) {
		int ret = Integer.MAX_VALUE;
		try {
			ret = Integer.parseInt(self);
		}
		catch (NumberFormatException nfe){}
		return ret;
    }
    
    /**
	 * The 'toReal' operation for strings
	 *
	 * @param self A String, on which the 'toReal' operation is called
	 * @return The real value of the given string if it represents a real number;
	 * otherwise Float.POSITIVE_INFINITY is returned
	 */
    public static float toReal(String self) {
		float ret = Float.POSITIVE_INFINITY;
		try {
			ret = Float.parseFloat(self);
		}
		catch (NumberFormatException nfe){}
		return ret;
    }
    
    /**
	 * The 'concat' operation for strings
	 *
	 * @param self A String, on which the 'concat' operation is called. May also be null (undefined)
	 * @param arg The argument of the 'concat' operation. May also be null (undefined)
	 * @return The concatenation <code>self+arg</code> as a string. If either argument is undefined (null),
	 * null is returned (as an Undefined marker)
	 */
    public static String concat(String self, String arg) {
		return self == null || arg == null ? null : self.concat(arg);
    }
    
    /**
	 * The 'contains' operation for strings
	 *
	 * @param self A String, on which the 'contains' operation is called
	 * @param arg The argument of the 'contains' operation
	 * @return true if <code>arg</code> is a substring of <code>self</code>. If either argument is undefined,
	 * undefined (false) is returned.
	 */
    public static boolean contains(String self, String arg) {
		return self == null || arg == null ? false : self.indexOf(arg) >= 0;
    }
    
    /**
	 * The 'pos' operation for strings.
	 *
	 * @param self A String, on which the 'pos' operation is called
	 * @param arg The argument of the 'pos' operation
	 * @return The position where the <code>arg</code> string begins in <code>self</code> or 0 if <code>arg</code> is not a
	 * substring in <code>self</code>. If either argument is undefined, Integer.MAX_VALUE is returned.
	 */
    public static int pos(String self, String arg) {
		return self == null || arg == null ? Integer.MAX_VALUE : self.indexOf(arg)+1;
    }
    
    /**
	 * The 'split' operation for strings
	 *
	 * @param self A String, on which the 'split' operation is called
	 * @param separators The argument of the 'split' operation
	 * @return the list (OclSequence) of tokens resulting by spliting the <code>self</code>
	 * String using the separators provided in <code>separators</code>.
	 */
    public static List split(String self, String separators) {
		if (self == null || separators == null)
			return null;
		
		StringTokenizer st = new StringTokenizer(self, separators, false);
		List ret = CollectionUtilities.newSequence();
		while (st.hasMoreTokens()) {
			ret.add(st.nextToken());
		}
		
		return ret;
    }
    
    /**
	 * The 'lt' operation for Strings
	 *
	 * @param self A String, on which the 'lt' operation is called
	 * @param arg The argument of the 'lt' operation
	 * @return The value <code>self.lt(arg)</code>
	 */
    public static boolean lt(String self, String arg) {
		return self == null ? false : self.compareTo(arg) < 0;
    }
    
    /**
	 * The 'gt' operation for Strings
	 *
	 * @param self A String, on which the 'gt' operation is called
	 * @param arg The argument of the 'gt' operation
	 * @return The value <code>self.gt(arg)</code>
	 */
    public static boolean gt(String self, String arg) {
		return self == null ? false : self.compareTo(arg) > 0;
    }
    
    /**
	 * The 'ge' operation for Strings (&gt;=)
	 *
	 * @param self A String, on which the 'ge' operation is called
	 * @param arg The argument of the 'ge' operation
	 * @return The value <code>self.ge(arg)</code>
	 */
    public static boolean ge(String self, String arg) {
		return self == null ? false : self.compareTo(arg) >= 0;
    }
    
    /**
	 * The 'le' operation for Strings
	 *
	 * @param self A String, on which the 'le' operation is called
	 * @param arg The argument of the 'le' operation
	 * @return The value <code>self.le(arg)</code>
	 */
    public static boolean le(String self, String arg) {
		return self == null ? false : self.compareTo(arg) <= 0;
    }
}
