/*
 * @(#)CollectionUtilities.java
 *
 * Copyright (c) 2003 Computer Science Research Laboratory,
 * "Babes-Bolyai" University, Cluj-Napoca, ROMANIA.
 * All Rights Reserved.
 */
package ro.ubbcluj.lci.codegen.framework.ocl;

import java.util.*;

import ro.ubbcluj.lci.codegen.framework.dt.OrderedSet;
import ro.ubbcluj.lci.codegen.framework.dt.Real;

/**
 * Contains the implementations of OCL collection operations. Loop operations are intentionally excluded.
 * <p>Notes: <li>All operations that have at least an OclAny argument are implemented using
 * class Object instead of OclAny; care must be taken when supplying primitive arguments (int, float, boolean)
 * because they must be wrapped. The Boolean, Integer and Real classes in package <code>dt</code>are designed
 * with this purpose.
 * <li>Undefined values are not always treated properly. Generally, the Undefined value in OCL is
 * equated with the Java null literal.</li></p>
 *
 * @author Cristian Botiza
 */
public final class CollectionUtilities {
    /**
	 * The 'size' operation. Works the same for all collection kinds.
	 *
	 * @param coll A Collection. May also be null (undefined)
	 * @return <code>Integer.MAX_VALUE</code> if the argument is undefined (null), or the
	 * real size of the collection otherwise.
	 */
    public static int size(Collection coll) {
		return coll != null ? coll.size() : Integer.MAX_VALUE;
    }
	
	/**
	 * The size of an object is defined as 1, if the object is defined and 0 otherwise. This
	 * operation was implemented because in OCL we may apply the 'size', 'is/not Empty'
	 * operations for instances also
	 *
	 * @param self The object whose size to compute. May also be null (undefined)
	 * @return 1 if <code>self</code> is defined and 0 otherwise
	 */
	public static int size(Object self) {
		return Ocl.isDefined(self) ? 1 : 0;
	}
    
    
    /**
	 * The 'includes(OclAny)' operation for generic OCL collections. The implementation returns false
	 * if the collection is undefined.
	 *
	 * @param coll A generic collection on which the 'includes' operation is called. May also be null
	 * @param o The argument of the 'includes' operation. May also be null
	 * @return false if <code>coll</code> is undefined or the value returned
	 * by the 'includes' test otherwise
	 */
    public static boolean includes(Collection coll, Object o) {
		return coll != null ? coll.contains(o) : false;
    }
    
    /**
	 * The 'excludes(OclAny)' operation for generic OCL collections. The implementation returns false
	 * if the collection is undefined.
	 *
	 * @param coll A generic collection on which the 'excludes' operation is called. May also be null
	 * @param o The argument of the 'excludes' operation. May also be null
	 * @return false if <code>coll</code> is undefined or the value returned
	 * by the 'excludes' test otherwise
	 */
    public static boolean excludes(Collection coll, Object o) {
		return coll != null ? !coll.contains(o) : false;
    }
    
    /**
	 * The 'count(OclAny)' operation for generic OCL collections. The implementation returns Undefined (Integer.
	 * MAX_VALUE) if the collection is undefined
	 *
	 * @param coll A generic collection, on which the 'count' operation is called. May also be null
	 * @param x The argument of the 'count' operation. May also be null
	 */
    public static int count(Collection coll, Object x) {
		boolean isUndefined = Ocl.isUndefined(coll);
		int c = Integer.MAX_VALUE;
		if (!isUndefined) {
			c = 0;
			Iterator it = coll.iterator();
			
			while (it.hasNext()) {
				Object current = it.next();
				if (current != null && current.equals(x) ||
					current == x) {
					c++;
				}
			}
		}
		return c;
    }
    
    /**
	 * The 'includesAll' operation for generic collections. The algorithm is duplicate-sensitive and works for
	 * all collection kinds (it properly implements the semantics specified in OCL).
	 *
	 * @param self The Collection on which the operation is called. May also be null
	 * @param arg The argument of the 'includesAll' operation. May also be null
	 * @return false If either <code>self</code> or <code>arg</code> is undefined, or the value
	 * of the 'includesAll' test otherwise
	 */
    public static boolean includesAll(Collection self, Collection arg) {
		if (arg == null || self == null) {
			return false;
		}
		Collection tmp2;
		try {
			//create a collection of the same kind with self
			tmp2 = (Collection)self.getClass().newInstance();
			tmp2.addAll(self);
			Iterator it = arg.iterator();
			boolean ia = true;//holds the result
			while (it.hasNext() && ia) {
				Object next = it.next();
				ia = tmp2.remove(next);
			}
			return ia;
		}
		catch (Exception ex) {
			System.err.println(ex.getMessage());
			return false;
		}
    }
    
    /**
	 * The 'excludesAll' operation for generic collections.
	 *
	 * @param self the Collection on which the operation is called. May also be null
	 * @param arg The argument of the 'excludesAll' operation. May also be null
	 * @return false if either <code>self</code> or <code>arg</code> is undefined, or the value
	 * of the 'excludesAll' test otherwise
	 */
    public static boolean excludesAll(Collection self, Collection arg) {
		if (arg == null || self == null) {
			return false;//not a proper semantics...
		}
		Iterator it = arg.iterator();
		boolean ea = true;
		while (it.hasNext() && ea) {
			if (self.contains(it.next())) {
				ea = false;
			}
		}
		return ea;
    }
    
    /**
	 * The 'isEmpty' operation for generic OCL collections. The implementation returns false
	 * if the collection is undefined.
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @return false if the argument is undefined or the value of the test <code>self-&gt;isEmpty()</code>
	 * otherwise
	 */
    public static boolean isEmpty(Collection self) {
		return self != null ? self.isEmpty() : false;
    }
	
	/**
	 * Generalized 'isEmpty' operation. This is implemented because in OCL we may have
	 * 'isEmpty' called for instances also
	 *
	 * @param self Any object
	 * @return The same as <code>self-&gt;isEmpty</code> (or <code>self-&gt;isUndefined</code>)
	 */
	public static boolean isEmpty(Object self) {
		return Ocl.isUndefined(self);
	}
    
	/**
	 * The 'notEmpty' operation for generic OCL collections. The implementation returns false
	 * if the collection is undefined.
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @return false if the collection is undefined or the value of the test <code>self-&gt;notEmpty</code>
	 * otherwise
	 */
    public static boolean notEmpty(Collection self) {
		return self != null ? !self.isEmpty() : false;
    }
	
	/**
	 * Generalized 'notEmpty' operation. This is implemented because in OCL we may have
	 * 'notEmpty' called for instances also
	 *
	 * @param self Any object
	 * @return The same as <code>self.isDefined</code>
	 */
	public static boolean notEmpty(Object self) {
		return Ocl.isDefined(self);
	}
    
    /**
	 * The 'sum' operation for generic OCL collection.
	 *
	 * @param self The collection on which the operation is called
	 * @return The sum of the elements in the collection, as a float
	 * @throws IllegalArgumentException If the collection does not contain descendants of Real
	 */
    public static float sum(Collection self) throws IllegalArgumentException {
		if (self == null) {
			return Float.POSITIVE_INFINITY;
		}
		float s = 0.0f;
		Iterator it = self.iterator();
		
		while (it.hasNext()) {
			Object next = it.next();
			if (!(next instanceof Real)) {
				throw new IllegalArgumentException("'sum()' expects only numeric objects. Invalid collection element("+next+")");
			}
			s+= ((Real)next).asReal();
		}
		return s;
    }
    
    /**
	 * The 'including' operation for bags and sequences
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @param o The argument of the 'including' operation. May also be null (undefined)
	 * @return The bag/sequence obtained by appending the specified object at the end of the given bag/sequence
	 */
    public static List including(List self, Object o) {
		
		if (self == null) {
			return null;
		}
		
		List ret = newBag(self);
		ret.add(o);
		
		return ret;
    }
    
    /**
	 * The 'including' operation for sets
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @param o The argument of the 'including' operation. May not be null (in the case of sets, undefined values
	 * are not allowed)
	 * @return The set obtained by adding <code>o</code> to <code>self</code>
	 * @throws IllegalArgumentException If the argument <code>o</code> is undefined
	 */
    public static Set including(Set self, Object o) throws IllegalArgumentException {
		
		if (self == null) {
			return null;
		}
		
		if (o == null) {
			throw new IllegalArgumentException("Undefined values not allowed in sets");
		}
		
		Set ret = newSet(self);
		ret.add(o);
		
		return ret;
    }
    
    /**
	 * The 'including' operation for OrderedSets
	 *
	 * @param self The ordered set on which the operation is called. May also be null
	 * @param o The argument of the 'including' operation. May not be null (undefined values are not allowed
	 * in sets)
	 * @return An ordered set obtained by adding <code>o</code> to <code>self</code>
	 * @throws IllegalArgumentException If the argument <code>o</code> is undefined
	 */
    public static OrderedSet including(OrderedSet self, Object o) throws IllegalArgumentException {
		
		if (self == null) {
			return null;
		}
		
		if (o == null) {
			throw new IllegalArgumentException("Undefined values not allowed in ordered sets");
		}
		
		OrderedSet ret = newOrderedSet(self);
		ret.add(o);
		
		return ret;
    }
    
    /**
	 * The 'excluding' operation for sequences/bags
	 *
	 * @param self The bag/sequence on which the operation is called. May also be null
	 * @param o The argument of the 'excluding' operation. May also be null
	 * @return null If the collection is undefined (null) or the bag/sequence obtained
	 * by appending <code>o</code> to <code>self</code> otherwise
	 */
    public static List excluding(List self, Object o) {
		if (self == null) {
			return null;
		}
		//do not create new objects if not necessary
		if (!self.contains(o)) {
			return self;
		}
		
		List ret = newBag(self);
		ret.remove(o);
		
		return ret;
    }
    
    /**
	 * The 'excluding' operation for sets
	 *
	 * @param self The set on which the operation is called. May also be null
	 * @param o The argument of the 'excluding' operation.
	 * @return The set obtained by removing <code>o</code> from </code>
	 */
    public static Set excluding(Set self, Object o) {
		if (self == null) {
			return null;
		}
		
		if (!self.contains(o)) {
			return self;
		}
		
		Set ret = newSet(self);
		ret.remove(o);
		
		return ret;
    }
    
    /**
	 * The 'excluding' operation for ordered sets
	 *
	 * @param self The ordered set on which the operation is called. May also be null
	 * @param o The argument of the 'excluding' operation.
	 * @return The ordered set obtained by removing <code>o</code> from <code>self</code>
	 */
    public static OrderedSet excluding(OrderedSet self, Object o) {
		if (self == null) {
			return null;
		}
		
		if (!self.contains(o)) {
			return self;
		}
		
		OrderedSet ret = newOrderedSet(self);
		ret.remove(o);
		
		return ret;
    }
    
    /**
	 * The 'flatten' operation for sets.
	 *
	 * @param self The set on which the operation is called. May also be null
	 * @return The flattened set obtained from <code>self</code>
	 */
    public static Set flatten(Set self) {
		if (self == null) {
			return null;
		}
		
		Set result = newSet();
		flatten(result, self);
		
		return result;
    }
    
	/**
	 * The 'flatten' operation for ordered sets
	 *
	 * @param self The ordered set on which the operation is called. May also be null
	 * @return The flattened ordered set obtained from <code>self</code>
	 */
    public static OrderedSet flatten(OrderedSet self) {
		if (self == null) {
			return null;
		}
		
		OrderedSet result = newOrderedSet();
		flatten(result, self);
		
		return result;
    }
    
    /**
	 * The 'flatten' operation for bags and sequences
	 *
	 * @param self The bag/sequence on which the operation is called
	 * @return The falttened bag/sequence obtained from <code>self</code>
	 */
    public static List flatten(List self) {
		if (self == null) {
			return null;
		}
		
		List result = newBag();//here it doesn't matter if we use newBag() or newSequence()
		flatten(result, self);
		
		return result;
    }
    
	/**
	 * The 'asSet' operation for generic OCL collections
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @return null if <code>self</code> is undefined or the value <code>self-&gt;asSet()</code>
	 */
    public static Set asSet(Collection self) {
		if (self == null) {
			return null;
		}
		
		//for efficiency
		if (self instanceof Set) {
			return (Set)self;
		}
		
		Set ret = newSet(self);
		return ret;
    }
    
	/**
	 * The 'asBag' operation for generic OCL collections
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @return null if <code>self</code> is undefined or the value <code>self-&gt;asBag()</code>
	 */
    public static List asBag(Collection self) {
		if (self == null) {
			return null;
		}
		
		if (self instanceof List) {
			return (List)self;
		}
		
		return newBag(self);
    }
    
	/**
	 * The 'asSequence' operation for generic OCL collections
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @return null if <code>self</code> is undefined or the value <code>self-&gt;asSequence()</code>
	 */
    public static List asSequence(Collection self) {
		return asBag(self);
    }
    
	/**
	 * The 'asOrderedSet' operation for generic OCL collections
	 *
	 * @param self The collection on which the operation is called. May also be null
	 * @return null if <code>self</code> is undefined or the value <code>self-&gt;asOrderedSet()</code>
	 */
    public static OrderedSet asOrderedSet(Collection self) {
		if (self == null) {
			return null;
		}
		
		if (self instanceof OrderedSet) {
			return (OrderedSet)self;
		}
		
		return newOrderedSet(self);
    }
    
    /**
	 * The 'union' operation for sets.
	 *
	 * @param self The set on which the operation is called
	 * @param arg Union argument collection
	 * @return A Set containing the union of the two sets. Duplicate elements are removed from the returned set.
	 * If either <code>self</code> or <code>arg</code> is undefined, null is returned
	 */
    public static Set union(Set self, Set arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		Set ret = newSet(self);
		ret.addAll(arg);
		
		return ret;
    }
    
    /**
	 * The 'union' operation for bags and sequences.
	 *
	 * @param self The bag / sequence on which the operation is called
	 * @param arg Union argument collection
	 * @return A Bag/Sequence containing the union of the two collections.
	 * If either <code>self</code> or <code>arg</code> is undefined, null is returned
	 */
    public static List union(List self, Set arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		List result = newBag(self);
		result.addAll(arg);
		return result;
    }
    
	/**
	 * The 'union' operation for a set and a bag/sequence
	 *
	 * @param self The set on which the operation is called. May also be null
	 * @param arg Union argument collection
	 * @return A Bag containing the union of the two collections
	 */
    public static List union(Set self, List arg) {
		return union(arg, self);
    }
    
	/**
	 * The 'union' operation for two bags/sequences
	 *
	 * @param self A bag/sequence on which the operation is called. May also be null
	 * @param arg Union argument collection
	 * @return null if either <code>self</code> is <code>arg</code> is undefined, or the union
	 * of the two collections otherwise
	 */
    public static List union(List self, List arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		List ret = newBag(self);
		ret.addAll(arg);
		return ret;
    }

    /**
     * The 'union' operation for an Object and a Set, in case the -&gt;union operation is called on an instance
     *
     * @param self An instance
     * @param arg The argument set
     * @return Undefined if self is Undefined or arg is Undefined, the union otherwise
     */
    public static Set union(Object self, Set arg) {
        Set result = self != null && arg != null ? newSet() : null;
        if (result != null) {
            result.add(self);
            result.addAll(arg);
        }
        return result;
    }

    /**
     * The 'union' operation for an Object and a Bag/Sequence, in case the -&gt;union operation is called on
     * an instance
     *
     * @param self An instance
     * @param arg The argument Bag/Sequence
     * @return Undefined if self.isUndefined or arg.isUndefined or the union otherwise
     */
    public static List union(Object self, List arg) {
        List result = self != null && arg != null ? newBag() : null;
        if (result != null) {
            result.add(self);
            result.addAll(arg);
        }
        return result;
    }

    /**
	 * The 'intersection' operation for bags/sequences.
	 *
	 * @param self The bag/sequence on which the operation is called. May also be null
	 * @param arg Intersection argument
	 * @return The intersection of the two collections, or null if one of them is undefined
	 */
    public static List intersection(List self, List arg) {
		if (self == null || arg == null) {
			return null;
		}
		//this operation is not allowed for ordered collections (see OCL spec)
		if (self instanceof OrderedSet || arg instanceof OrderedSet) {
			return null;
		}
		
		List ret = newBag(self);
		ret.retainAll(arg);
		return ret;
    }
    
    /**
	 * The 'intersection' operation for sets.
	 *
	 * @param self The set on which the operation is called. May also be null
	 * @param arg Intersection argument
	 * @return null if one of the two sets is undefined, or the intersection of the two sets otherwise
	 */
    public static Set intersection(Set self, Set arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		//ordered collections not allowed for this operation (See OCL spec)
		if (self instanceof OrderedSet || arg instanceof OrderedSet) {
			return null;
		}
		Set ret = newSet(self);
		ret.retainAll(arg);
		return ret;
    }
    
    /**
	 * The 'intersection' operation for sets and bags.
	 *
	 * @param self The bag/sequence on which the operation is called. May also be null
	 * @param arg The argument set
	 * @return null if either of the two collections is undefined, or the intersection otherwise.
	 * The result is a sequence, since one collection is a sequence
	 */
    public static List intersection(List self, Set arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		//ordered collections not allowed
		if (self instanceof OrderedSet || arg instanceof OrderedSet) {
			return null;
		}
		
		List ret = newBag(self);
		ret.retainAll(arg);
		
		return ret;
    }
    
    /**
	 * The 'intersection' operation for sets and bags.
	 *
	 * @param self The set on which the operation is called. May also be null
	 * @param arg Intersection argument
	 */
    public static List intersection(Set self, List arg) {
		return intersection(arg, self);
    }
    
    
    /**
	 * The '-' operation for sets.
	 *
	 * @param self The set on which the operation is called. May also be null.
	 * @param arg Operation argument. May also be null.
	 * @return null if either of the two sets is undefined, or the result of the minus
	 * operation otherwise
	 */
    public static Set minus(Set self, Set arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		Set ret = newSet();
		Iterator it = self.iterator();
		while (it.hasNext()) {
			Object candidate = it.next();
			if (!arg.contains(candidate))
				ret.add(candidate);
		}
		
		return ret;
		
    }
    
    /**
	 * The '-' operation for OrderedSets.
	 *
	 * @param self The ordered set on which the operation is called. May also be null.
	 * @param arg Operation argument. May also be null.
	 * @return null if either of the two sets is undefined, or the result of the minus
	 * operation otherwise
	 */
    public static OrderedSet minus(OrderedSet self, Set arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		OrderedSet ret = newOrderedSet();
		Iterator it = self.iterator();
		while (it.hasNext()) {
			Object c = it.next();
			if (!arg.contains(c))
				ret.add(c);
		}
		
		return ret;
    }
    
    /**
	 * The 'symmetricDifference' operation, only for OCL sets.
	 *
	 * @param self The set on which the operation is called. May also be null.
	 * @param arg Argument set
	 * @return The symmetricDifference of the two sets or null if one of them is undefined
	 */
    public static Set symmetricDifference(Set self, Set arg) {
		if (self == null || arg == null || self instanceof OrderedSet || arg instanceof OrderedSet) {
			return null;
		}
		/*
		 A.symmetricDifference(B) = (A+B)-(A.intersection(B)) =>
		 the following is a very simple and efficient algorithm:
		 Set result = newSet();
		 result.addAll(self);
		 foreach e in arg {
		 if result.contains(e) then result= result-{e}
		 else result = result+{e}
		 }*/
		Set result = newSet();
		result.addAll(self);
		Iterator it = arg.iterator();
		while (it.hasNext()) {
			Object next = it.next();
			if (!result.add(next)) {
				result.remove(next);
			}
		}
		return result;
    }
    
    /**
	 * The 'append' operation for sequences.
	 *
	 * @param self The sequence on which the operation is called. May also be null.
	 * @param arg Object to append. May also be null.
	 * @return The sequence obtained by appending <code>o</code> to <code>self</code> or null
	 * if self is undefined
	 */
    public static List append(List self, Object arg) {
		if (self == null /*|| arg == null*/) {
			return null;
		}
		
		List ret = newSequence();
		append(ret, self, arg);
		return ret;
    }
    
    /**
	 * The 'append' operation for OrderedSets.
	 *
	 * @param self The ordered on which the operation is called. May also be null.
	 * @param arg Object to append. May also be null.
	 * @return The ordered set obtained by appending <code>o</code> to <code>self</code> or null
	 * if self or o is undefined
	 */
    public static OrderedSet append(OrderedSet self, Object arg) {
		if (self == null || arg == null) {
			return null;//do not allow undefined values in an ordered set
		}
		
		OrderedSet ret = newOrderedSet();
		
		append(ret, self, arg);
		return ret;
    }
    
    /**
	 * The 'prepend' operation for sequences. Due to our conventions, it also works fine for bags (though it shouldn't).
	 *
	 * @param self The sequence on which the operation is called. May also be null.
	 * @param arg Object to prepend. May also be null.
	 * @return The ordered set obtained by prepending <code>o</code> to <code>self</code> or null
	 * if self is undefined
	 */
    public static List prepend(List self, Object arg) {
		if (self == null /*|| arg == null*/) {
			return null;
		}
		
		List ret = newSequence();
		ret.add(arg);
		ret.addAll(self);
		
		return ret;
    }
    
    /**
	 * The 'prepend' operation for ordered sets.
	 *
	 * @param self The ordered set on which the operation is called. May also be null.
	 * @param arg Object to prepend. May also be null
	 * @return The ordered set obtained by prepending <code>arg</code> to <code>self</code>
	 */
    public static OrderedSet prepend(OrderedSet self, Object arg) {
		if (self == null || arg == null) {
			return null;
		}
		
		OrderedSet ret = newOrderedSet();
		
		ret.add(arg);
		ret.addAll(self);
		
		return ret;
    }
    
    /**
	 * The 'insertAt' operation for sequences. Due to our conventions, it also works fine for bags (though it shouldn't).
	 *
	 * @param self The sequence on which the operation is called. May also be null.
	 * @param idx The index where to insert, 1 based
	 * @param arg Object to insert. May also be null.
	 * @return The sequence obtained by inserting <code>arg</code> at index <code>idx</code> or
	 * null if <code>self</code> is undefined or <code>idx</code> is out of the admissible bounds
	 */
    public static List insertAt(List self, int idx, Object arg) {
		if (self == null || idx < 0 || idx > self.size()) {
			return null;//undefined
		}
		
		List ret = newBag();
		ret.addAll(self);
		ret.add(idx, arg);
		
		return ret;
		
    }
    
    /**
	 * The 'insertAt' operation for OrderedSets.
	 *
	 * @param self The ordered set on which the operation is called. May also be null.
	 * @param idx The index where to insert, 1 based
	 * @param arg Object to insert. May also be null.
	 * @return The ordered set obtained by inserting <code>arg</code> at index <code>idx</code> or
	 * null if <code>self</code> is undefined or <code>idx</code> is out of the admissible bounds
	 */
    public static OrderedSet insertAt(OrderedSet self, int idx, Object arg) {
		if (self == null || idx < 0 || idx > self.size() || arg == null) {
			return null;//undefined
		}
		
		OrderedSet ret = newOrderedSet();
		
		ret.addAll(self);
		ret.add(idx, arg);
		
		return ret;
		
    }
    
    /**
	 * Returns the subsequence start..end from the given sequence. The resulting sequence
	 * is empty if <code>start &lt end</code>.
	 *
	 * @param self The ordered collection on which the operation is called
	 * @param start 1-based start index for subsequence
	 * @param end 1-based end index for subsequence
	 * @return The ordered collection with the elements self-&gt;at(start), ..., self-&gt;at(end)
	 */
    public static List subSequence(List self, int start, int end) {
		if (self == null || end > self.size() || start < 1) {
			return null;//The result is Undefined
		}
		
		List ret = newSequence();
		int i;
		for (i = start-1; i<end; i++) {//start is 1-based, as in OCL
			ret.add(self.get(i));
		}
		return ret;
    }
    
    /**
	 * Returns the sub ordered set start..end from the given ordered set. The resulting ordered set
	 * is empty if <code>start &lt end</code>.
	 *
	 * @param self The ordered set on which the operation is called
	 * @param start 1-based start index for sub ordered set
	 * @param end 1-based end index for sub ordered set
	 * @return The ordered set with the elements self-&gt;at(start), ..., self-&gt;at(end) or null
	 * if <code>self</code> is undefined or start&gt;end
	 */
    public static OrderedSet subOrderedSet(OrderedSet self, int start, int end) {
		if (self == null || start < 1 || end > self.size()) {
			return null;
		}
		
		OrderedSet ret = newOrderedSet();
		for (int i = start-1; i<end; i++) {
			ret.add(self.get(i));
		}
		
		return ret;
    }
    
    /**
	 * The 'at' operation. The index given as parameter is 1-based (as in OCL)
	 *
	 * @param self The sequence on which the operation is called. May also be null.
	 * @param idx 1 based index
	 * @return null if self is undefined or idx is out of the admissible bounds, or the
	 * object at the specified position otherwise
	 */
    public static Object at(List self, int idx) {
		if (self == null) {
			return null;
		}
		Object r;
		try {
			r = self.get(idx-1);
		}
		catch (IndexOutOfBoundsException e) {
			System.err.println(e.getMessage());
			r = null;
		}
		return r;
    }
    
    /**
	 * The 'indexOf' operation. The returned index is 1-based as in OCL. Because both Sequences and OrderedSets
	 * are modeled using Java lists, there is only one implementation method, having a Java list argument.
	 *
	 * @param self The ordered collection on which the operation is called. May also be null.
	 * @param arg Object to search for. May also be null.
	 * @return The 1-based index of the first occurence of the <code>arg</code> object or <code>Integer.MAX_VALUE</code>
	 * if no such object is found
	 */
    public static int indexOf(List self, Object arg) {
		if (self == null) {
			return Integer.MAX_VALUE;
		}
		int i = self.indexOf(arg);
		return i<0 ? Integer.MAX_VALUE : i+1;
    }
    
    /**
	 * The 'first' operation, defined for Sequences and OrderedSets. Because both collection kinds
	 * are implemented using Java lists, only one method is supplied, taking a List as argument.
	 * When reading the return value, unwrapping might be necessary.
	 *
	 * @param self The ordered collection on which the operation is called. May also be null.
	 * @return null if self is undefined or empty or the first element in it otherwise
	 */
    public static Object first(List self) {
		if (self != null && self.size() > 0) {
			return self.get(0);
		}
		return null;
    }
    
	/**
	 * The 'last' operation for ordered collections.
	 *
	 * @param self The ordered collection on which the operation is called. May also be null.
	 * @return null if self is undefined or empty or its last element otherwise
	 */
    public static Object last(List self) {
		if (self != null && self.size() > 0) {
			return self.get(self.size()-1);
		}
		return null;
    }
    
    public static void add(Collection c, Object x) {
		if (c != null) {
			c.add(x);
		}
    }
    
	//--------factory methods--------------
	
	/**
	 * Creates a new OCL set
	 *
	 * @return A new, empty set
	 */
    public static Set newSet() {
		return new LinkedHashSet();
    }
    
	/**
	 * Creates a new set considering the elements in the supplied collection
	 *
	 * @param c Collection to copy from. May also be null, in which case an empty set will be created.
	 * @return A set containing only the distinct elements in <code>c</code> or the empty set if
	 * c is undefined (null)
	 */
    public static Set newSet(Collection c) {
		return c != null ? new LinkedHashSet(c) : new LinkedHashSet();
    }
    
	/**
	 * Creates a new ordered set
	 *
	 * @return A new, empty ordered set
	 */
    public static OrderedSet newOrderedSet() {
		return new OrderedSet();
    }
    
	/**
	 * Creates a new ordered set considering the elements in the specified collection
	 *
	 * @param c Collection to copy from. May also be null.
	 * @return An ordered set containing only the distinct elements in <code>c</code> or the
	 * empty ordered set if c is null
	 */
    public static OrderedSet newOrderedSet(Collection c) {
		return c != null ? new OrderedSet(c) : new OrderedSet();
    }
    
	/**
	 * Creates a new bag
	 *
	 * @return A new, empty bag
	 */
    public static List newBag() {
		return new ArrayList();
    }
    
	/**
	 * Creates a new bag containing the elements in the specified collection. The appearance order may not
	 * be preserved (no explicit guarantee provided)
	 *
	 * @param c Collection to copy from. May also be null.
	 * @return A new bag with the elements in <code>c</code> or an empty bag if c is undefined (null)
	 */
    public static List newBag(Collection c) {
		return c != null ? new ArrayList(c) : new ArrayList();
    }
    
	/**
	 * Creates a new sequence
	 *
	 * @return A new, empty sequence
	 */
    public static List newSequence() {
		return new ArrayList();
    }
    
	/**
	 * Creates a new sequence with the elements from the specified collection. The appearance order may not be preserved
	 * (no explicit guarantee provided).
	 *
	 * @param c Collection to copy from. May also be null.
	 * @return A sequence containing the elements in <code>c</code> or an empty sequence if c is null
	 */
    public static List newSequence(Collection c) {
		return c != null ? new ArrayList(c) : new ArrayList();
    }
    
    /**
	 * Registers all the elements in the range <code>start..stop</code> in the
	 * specified collection. Due to Java dynamic binding, the implementation is
	 * collection-kind sensitive.
	 *
	 * @param arg Collection to modify. May also be null.
	 * @param start
	 * @param stop
	 */
    public static void addRange(Collection arg, int start, int stop) {
		if (arg != null ) {
			for (int i = start; i<stop; i++) {
				arg.add(new ro.ubbcluj.lci.codegen.framework.dt.Integer(i));
			}
		}
    }
	
	//--------------utility methods-----------------
    
	/**
	 * Flatten a non-null collection
	 *
	 * @param result Accumulator collection. Must not be null
	 * @param arg The initial collection, non-null
	 */
    private static void flatten(Collection result, Collection arg) {
		//assume arg is not null
		try {
			Iterator it = arg.iterator();
			while (it.hasNext()) {
				Object next = it.next();
				if (!(next instanceof Collection)) {
					result.add(next);
				}
				else {
					Collection temp = (Collection)result.getClass().newInstance();
					flatten(temp, (Collection)next);
					result.addAll(temp);
				}
			}
		}
		catch (Exception e) {
			System.err.println(e.getMessage());
		}
    }
    
	/**
	 * Appends an element to the specified collection
	 *
	 * @param result A non-null collection. Its initial contents is lost
	 * @param target Collection to start from, not null
	 * @param x Object to append. May also be null.
	 */
    private static void append(Collection result, Collection target, Object x) {
		result.clear();
		result.addAll(target);
		result.add(x);
    }
}
