/*
 * @(#)OclType.java
 *
 * Copyright (c) 2003 Computer Science Research Laboratory,
 * "Babes-Bolyai" University, Cluj-Napoca, ROMANIA.
 * All Rights Reserved.
 */
package ro.ubbcluj.lci.codegen.framework.ocl;
import java.util.*;

import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import ro.ubbcluj.lci.codegen.framework.dt.OrderedSet;
import ro.ubbcluj.lci.codegen.framework.dt.Real;

/**
 * Models the OclType metaclass. This class is also responsible for RunTime instance management
 * as required by the 'allInstances()' operation.
 * <p>The implementation exclusively uses Java reflection.
 * As a consequence, some operations are not properly implemented in this version.</p>
 *
 * @author Cristian Botiza
 */
public final class OclType {
	
    private static final List oclAnyOps, oclAnyAttrs, oclAnyAssEnds;
	
    /**
	 * Used to manage the 'allInstances' operation
	 * <p>key: java.lang.Class</p>
	 * <p>value: java.util.List - the set of registered 'instances' for the class given as key;
	 * such a list contains WeakReference objects and is updated internally</p>
	 */
    private static final HashMap instances;
	
    static {
		
		instances = new HashMap();
		
		oclAnyOps = new ArrayList();
		oclAnyOps.add("oclIsKindOf");
		oclAnyOps.add("oclIsTypeOf");
		oclAnyOps.add("oclAsType");
		oclAnyOps.add("isUndefined");
		oclAnyOps.add("isDefined");
		oclAnyOps.add("oclInState");
		oclAnyOps.add("oclIsNew");
		
		oclAnyAttrs = Collections.EMPTY_LIST;
		
		oclAnyAssEnds = new ArrayList();
		oclAnyAssEnds.add("oclType");
    }
	
    /**
	 * Constructs a simple (non-collection) type.
	 *
	 * @param rt The encapsulated class
	 */
    public OclType(Class rt) {
		realType = TypeUtilities.normalize(rt);
    }
	
    /**
	 * Constructs an OclType such as <code>Set(Set(Integer))</code>.
	 *
	 * @param types An ordered list of types such that all but the last are Collection descendants.
	 * Particularly, if <code>types.length==1</code>, then <code>types[0]</code> must
	 * not be a collection descendant; moreover, in this case this constructor
	 * is equivalent with the constructor with non-vector argument. The types after
	 * the first non-collection descendant are simply ignored.
	 *
	 * @throws IllegalArgumentException If all the types in the argument vector are Collection descendants
	 */
    public OclType(Class[] types) {
		int i, l = types.length-1;
		OclType current = this;
		for (i=0; i<l; i++) {
			if (!Collection.class.isAssignableFrom(types[i])) break;
			
			current.realType = TypeUtilities.normalize(types[i]);
			OclType temp = new OclType();
			current.innerType = temp;
			current = temp;
		}
		if (i >= types.length) {
			throw new IllegalArgumentException("Inner type not specified for collection type");
		}
		current.realType = types[i];
    }
	
    private OclType() {
    }
	
    /**
	 * For a non collection type, returns the type itself; for a collection type
	 * returns the corresponding Class for the collection kind
	 *
	 * @return The real type
	 */
    public Class getRealType() {
		return realType;
    }
	
    /**
	 * Retrieves the inner type for collection types.
	 *
	 * @return The inner type for collection types or null otherwise
	 */
    public OclType getInnerType() {
		return innerType;
    }
	
    public void setInnerType(OclType it) {
		innerType = it;
    }
	
    /**
	 * Tests whether this type is a collection type
	 *
	 * @return true if this is a collection type, false otherwise
	 */
    public boolean isCollectionType() {
		return Collection.class.isAssignableFrom(realType);
    }
	
    public boolean equals(Object o) {
		if (!(o instanceof OclType)) {
			return false;
		}
		
		OclType it = (OclType)o, is = this;
		
		while (is != null) {
			if (it == null) {
				return false;
			}
			if (it.realType != is.realType) {
				return false;
			}
			is = is.innerType;
			it = it.innerType;
		}
		return true;
    }
	
    /**
	 * Tests if this type is a descendant of the given type or the two types are equal
	 * (More formally, this type conforms to <code>t</code>)
	 *
	 * @param t A type object. May also be null.
	 * @return true if <code>this</code> conforms to <code>t</code>
	 */
    public boolean conforms(OclType t) {
		OclType child = this, pp = t;
		while (child != null) {
			if (pp == null) {
				return false;
			}
			if (!pp.realType.isAssignableFrom(child.realType)) {
				return false;
			}
			pp = pp.innerType;
			child = child.innerType;
		}
		return true;
    }
	
    /**
	 * Retrieves the common supertype of the given type
	 *
	 * @param types A vector of type objects
	 */
    public static OclType commonType(OclType[] types) {
		int n = types.length, i;
		if (n < 1) {
			return new OclType(Object.class);
		}
		OclType temp = types[0];
		for (i=1; i<n; i++) {
			temp = commonType(temp, types[i]);
		}
		return temp;
    }
	
    /**
	 * Retrieves the common type of the two given types.
	 *
	 * Note: Due to the way we treat collections in this version, the interface
	 * {@link java.util.Collection} may be encountered as a common parent. This should be considered
	 * the OclCollection OCL type. As a BUG, there is no way to distinguish between OclBag and OclSequence,
	 * which are both modled with the {@link java.util.List} interface;
	 * As a consequence the common parent for these two types will be List, not (Ocl)Collection as expected
	 *
	 * @param t1 A type object. Should not be null
	 * @param t2 A type object. Should not be null
	 * @return The common supertype of the two types
	 */
    static final OclType commonType(OclType t1, OclType t2) {
		boolean bIsColl1 = t1.isCollectionType(), bIsColl2 = t2.isCollectionType();
		//if one type is a collection type and the other is not, the common
		//parent is OclAny (Object)
		if (bIsColl1 ^ bIsColl2) {
			return new OclType(Object.class);
		}
		Class start = commonType(t1.getRealType(), t2.getRealType());
		
		//if both are simple types:
		if (!bIsColl1 && !bIsColl2) {
			return new OclType(start);
		}
		
		//else, if both are collection types:
		OclType ciType = commonType(t1.innerType, t2.innerType), ret;
		ret = new OclType(start);
		ret.setInnerType(ciType);
		return ret;
    }
	
    /**
	 * The 'name()' standard operation.
	 *
	 * @return The name of this type, such as <code>Set<code>,
	 * <code>List(Integer)<code> and so on. Particularly, Bags and Sequences are
	 * specified as 'List'
	 */
    public String name() {
		StringBuffer bf = new StringBuffer();
		Class temp = TypeUtilities.normalize(realType);
		String baseName = BasicUtilities.getLastComponent(temp.getName());
		//some adaptations are necessary:
		if (OrderedSet.class == temp) {
			baseName = "OrderedSet";
		}
		else
			if (List.class.isAssignableFrom(temp)) {
				baseName = "List";//could be Bag/Sequence
			}
			else if (Set.class.isAssignableFrom(temp)) {
					baseName = "Set";
				}
		
		bf.append(baseName);
		
		//treat inner types if the case:
		if (isCollectionType()) {
			String inner = innerType.name();
			bf.append('(').append(inner).append(')');
		}
		
		return bf.toString();
    }
	
    /**
	 * Retrieves the Class object representing the common parent of the two given
	 * classes. 'Extends' as well as 'implements' relationships are taken into account.
	 *
	 * @param c1 A non-null class object
	 * @param c2 A non-null class object
	 * @return The class object that is the common parent of the two given class
	 * objects. This may as well be an interface
	 */
    static final Class commonType(Class c1, Class c2) {
		//the c1Par set holds the parent classes/interfaces of c1,
		//c2Par does the same for c2
		
		//use an ordered HashSet implementation, to ensure that the first parent is obtained
		HashSet c1Par = new LinkedHashSet(), c2Par = new LinkedHashSet(), commonParents = null, temp,
			c1GlobalPar = new LinkedHashSet(), c2GlobalPar = new LinkedHashSet();
		
		c1Par.add(c1);
		c2Par.add(c2);
		c1GlobalPar.add(c1);
		c2GlobalPar.add(c2);
		
		boolean ready = false;
		int i;
		while (!ready) {
			
			//1: determine the intersection
			commonParents = (HashSet)c1GlobalPar.clone();
			commonParents.retainAll(c2GlobalPar);
			ready = !commonParents.isEmpty();
			
			
			//2: Update the two sets, to include the parent classes and exclude the old values
			Iterator it;
			Class c;
			temp = (HashSet)c1Par.clone();
			
			c1Par.clear();
			for (it = temp.iterator(); it.hasNext();) {
				c = (Class)it.next();
				//the call below places Object.class on the last position in the array (if Object.class is a direct parent)
				Class[] p = TypeUtilities.parents(c);
				for (i=0; i<p.length; i++) {
					c1Par.add(p[i]);
				}
			}
			c1GlobalPar.addAll(c1Par);
			
			temp = (HashSet)c2Par.clone();
			c2Par.clear();
			for (it = temp.iterator(); it.hasNext();) {
				c = (Class)it.next();
				Class[] p = TypeUtilities.parents(c);
				for (i=0; i<p.length; i++) {
					c2Par.add(p[i]);
				}
			}
			c2GlobalPar.addAll(c2Par);
		}
		return (Class)commonParents.iterator().next();
    }
	
    //--------------some standard operations prescribed by the specification------
    //--------------not all are properly implemented--------------------------------------
	
    /**
	 * The 'operations()' standard operation. Retrives a set with the distinct names
	 * of all the operations of this type. The method artificially inserts the names of the
	 * operations in OclAny.
	 *
	 * @return A Set containing the names of all operations defined in this type, either
	 * declared or inherited. No assumption is made about the visibility of any operation
	 */
    public Set operations() {
		Set result = CollectionUtilities.newSet();
		int i;
		ArrayList classes = new ArrayList(); //acts as a queue
		classes.add(realType);
		
		while (!classes.isEmpty()) {
			Class current = (Class)classes.remove(0);
			
			//add the names of all methods from current class
			Method[] m = current.getDeclaredMethods();
			for (i=0; i<m.length; i++)
				result.add(m[i].getName());
			
			//register all parent classes/interfaces
			if (current.getSuperclass() != null) {
				classes.add(current.getSuperclass());
			}
			/*operations from interfaces are not required, because if this class is
			 *already compiled, it implements all those operations, so the names will
			 *be retrieved only by traversing the inheritance ('extends') tree. However,
			 *this is no true for abstract classes*/
			if (Modifier.isAbstract(current.getModifiers())) {
				Class[] iis = current.getInterfaces();//implemented interfaces
				for (i = 0; i<iis.length; i++) {
					classes.add(iis[i]);
				}
			}
		}
		
		//finally add the names of the operations inherited from OclAny
		result.addAll(oclAnyOps);
		
		return result;
    }
	
	
    /**
	 * Retrives a set with the names of all the attributes of this type.
	 * <p><i>This method is implemented using Java reflection. As such, there is no
	 * way to distinguish between attributes and association ends.</i></p>
	 *
	 * @return The set with the names of all attributes defined in this type, either declared
	 * in it or inherited. No tests are performed regarding the visibility of any (inherited)
	 * field. Moreover, because the result is a set, no duplicate names will appear, even
	 * if an attribute with the same name as a declared attribute is inherited from a
	 * supertype.
	 */
    public Set attributes() {
		Set result = CollectionUtilities.newSet();
		int i;
		
		ArrayList classes = new ArrayList(); //acts as a queue
		classes.add(realType);
		
		while (!classes.isEmpty()) {
			Class current = (Class)classes.remove(0);
			
			//add all attributes from current class
			Field[] f = current.getDeclaredFields();
			for (i=0; i<f.length; i++) {
				result.add(f[i].getName());
			}
			
			//register all parent classes/interfaces
			if (current.getSuperclass() != null) {
				classes.add(current.getSuperclass());
			}
			Class[] ints = current.getInterfaces();
			for (i=0; i<ints.length; i++) {
				classes.add(ints[i]);
			}
		}
		
		//finally add the names of any attributes inherited from OclAny:
		result.addAll(oclAnyAttrs);
		
		return result;
    }
	
    /**
	 * Retrieves a Set with the names of all the association ends of this type.
	 * This method is <b>not properly implemented</b>. See {@link #attributes()} for
	 * an explanation.
	 *
	 * @return A set containing only the names of the association ends inherited from OclAny
	 */
    public Set associationEnds() {
		Set r = CollectionUtilities.newSet();
		r.addAll(oclAnyAssEnds);
		
		return r;
    }
	
    /**
	 * Returns the set containing all direct supertypes of this type
	 *
	 * @return The set of all direct supertypes for this type
	 */
    public Set supertypes() {
		Set result = CollectionUtilities.newOrderedSet();
		Class[] parents = TypeUtilities.parents(realType);
		boolean isCollection = isCollectionType();
		
		//the parents of the encapsulated type
		if (isCollection) {
			Set innerSupertypes = innerType.supertypes();
			Iterator its = innerSupertypes.iterator();
			while (its.hasNext()) {
				OclType t = new OclType(realType);
				t.innerType = (OclType)its.next();
				result.add(t);
			}
		}
		
		for (int i=0; i<parents.length; i++) {
			OclType pt = new OclType(parents[i]);
			pt.innerType = isCollection && Collection.class.isAssignableFrom(parents[i]) ? innerType : null;
			result.add(pt);
		}
		
		return result;
    }
	
    /**
	 * Retrieves the set of all the supertypes for this type.
	 *
	 * @return The set containing all the supertypes for this type
	 */
    public Set allSupertypes() {
		Set result = CollectionUtilities.newOrderedSet();
		ArrayList temp = new ArrayList();
		temp.add(this);
		while (!temp.isEmpty()) {
			OclType currentType = (OclType)temp.remove(0);
			Set directParents = currentType.supertypes();
			temp.addAll(directParents);
			result.addAll(directParents);
		}
		return result;
    }
	
    public String toString() {
		return name();
    }
	
    /**
	 * The 'allInstances' operation on this type.
	 *
	 * @return A set containing all alive instances for this type.
	 */
    public Set allInstances() {
		return allInstances(realType);
    }
	
    /**
	 * The 'allInstances' operation. The implementation synchronizes on the given Class object.
	 * It also updates the instance list for the given type, by removing any WeakReference objects
	 * that do not point to alive objects any more.
	 *
	 * @param type The class whose instances are requested. Must not be null
	 * @return A Set containing all alive instances for the given type.
	 */
    private static Set allInstances(Class type) {
		Set result = CollectionUtilities.newSet();
		List instProxies = null;
		synchronized (type) {
			instProxies = (List)instances.get(type);
			if (instProxies != null) {
				List tmp = new ArrayList();//contains all identified invalid refs
				//build the result List
				System.gc();
				Iterator it = instProxies.iterator();
				while (it.hasNext()) {
					WeakReference ref = (WeakReference)it.next();
					
					if (ref.get() == null) {
						tmp.add(ref);
					}
					else {
						result.add(ref.get());
					}
				}
				instProxies.removeAll(tmp);
				//clear the map, if the case
				if (instProxies.isEmpty()) {
					instances.remove(type);
				}
				tmp = null;
			}
		}
		
		return result;
    }
	
    /**
	 * Enques the given instance in the list of instances for the given class. This method
	 * synchronizes on the given Class object, to ensure that no other thread modifies or
	 * reads the attached instance list while the current thread inserts the new instance
	 *
	 * @param inst Instance to be registered; should not be null
	 * @param cls The class in whose instance list should <code>inst</code> be added
	 */
    public static void registerInstance(Object inst, Class cls) {
		synchronized(cls) {
			List instProxies = (List)instances.get(cls);
			WeakReference newRef = new WeakReference(inst);
			if (instProxies == null) {
				instProxies = new ArrayList();
				instances.put(cls, instProxies);
			}
			instProxies.add(newRef);
		}
    }
	
    private Class realType;
    private OclType innerType = null;
}

final class TypeUtilities {
    static Class[] parents(Class c) {
		
		c = normalize(c);
		
		Class[] ints = c.getInterfaces();
		Class p = c.getSuperclass();
		//some tricks are required: if c is an interface and p is null
		//p should in fact be Object.class
		if (p == null && c != Object.class) {
			p = Object.class;
		}
		Class[] result = new Class[p!=null ? 1 + ints.length : ints.length];
		if (p != null) {
			//add extended class at the end, so that Object.class is the last parent
			result[result.length - 1] = p;
		}
		System.arraycopy(ints, 0, result, 0, ints.length);
		return result;
    }
	
    static Class normalize(Class c) {
		Class v = c;
		if (c == int.class || c == Integer.class) {
			v = ro.ubbcluj.lci.codegen.framework.dt.Integer.class;
		}
		else if (c == float.class || c == double.class || c == Float.class
				 || c == Double.class) {
				v = Real.class;
			}
			else if (c == boolean.class || c == Boolean.class) {
					v = ro.ubbcluj.lci.codegen.framework.dt.Boolean.class;
				}
		return v;
    }
}
