/*
 * @(#)Pairs.java
 *
 * Copyright (c) 2003 Computer Science Research Laboratory,
 * "Babes-Bolyai" University, Cluj-Napoca, ROMANIA.
 * All Rights Reserved.
 */
package ro.ubbcluj.lci.codegen.framework.ocl;
import java.util.*;

import ro.ubbcluj.lci.codegen.framework.dt.OrderedSet;

/**
 * Class Pairs. Instances of this class hold object pairs and sort them by the second
 * component. This class is used in the sortedBy() code template. A typical usage scenario
 * would do the following things:
 * <code><p>
 * Pairs pairList = new Pairs();</p>
 * <p>repeatedly add pairs, using the addPair() method;</p>
 * <p>pairList.sort()</p>
 * <p>use pairList.asOrderedSet() / pairList.asSequence()</p>
 * </p></code>
 *
 * @author Cristian Botiza
 */
public final class Pairs {
	
	public Pairs() {
		c = new DefaultComparator();
	}
	
	/**
	 * Registers a new pair.
	 *
	 * @param first The first component of the pair
	 * @param second The second component of the pair
	 */
	public void addPair(Object first, Object second) {
		Pair p = new Pair();
		p.firstComponent = first;
		p.secondComponent = second;
		synchronized(pairs) {
			pairs.add(p);
		}
	}
	
	/**
	 * Sort the pair list by the second component. If two objects do not implement
	 * {@link java.lang.Comparable}, they are considered equal and thus no sorting takes
	 * place on them.
	 */
	public void sort() {
		synchronized(pairs) {
			Collections.sort(pairs, c);
		}
	}
	
	/**
	 * Returns the sequence representation of the list of the first components of registered
	 * pairs.
	 *
	 * @return A sequence containing the first components of the registered pairs, in the order they were
	 * added
	 */
	public List asSequence() {
		List res = CollectionUtilities.newSequence();
		synchronized(pairs) {
			for (int i = 0, n = pairs.size(); i<n; i++) {
				res.add(((Pair)pairs.get(i)).firstComponent);
			}
		}
		return res;
	}
	
	/**
	 * Returns the OrderedSet representation of the list containing the first components
	 * of registered pairs.
	 *
	 * @return An ordered set containing the first components of any registered pairs, in the order they
	 * were added
	 */
	public OrderedSet asOrderedSet() {
		OrderedSet res = CollectionUtilities.newOrderedSet();
		synchronized(pairs) {
			for (int i = 0, n = pairs.size(); i<n; i++) {
				res.add(((Pair)pairs.get(i)).firstComponent);
			}
		}
		return res;
	}
	
	private static class Pair {
		Object firstComponent, secondComponent;
	}
	
	private static class DefaultComparator implements Comparator {
		
		/**
		 * Compares its two arguments for order.  Returns a negative integer,
		 * zero, or a positive integer as the first argument is less than, equal
		 * to, or greater than the second.<p>
		 *
		 * @param o1 the first object to be compared.
		 * @param o2 the second object to be compared.
		 * @return a negative integer, zero, or a positive integer as the
		 * 	       first argument is less than, equal to, or greater than the
		 *	       second.
		 * @throws ClassCastException if the arguments' types prevent them from
		 * 	       being compared by this Comparator.
		 */
		public int compare(Object o1, Object o2) {
			Pair p1 = (Pair)o1, p2 = (Pair)o2;
			//compares the second components of the two pairs:
			if (p1.secondComponent instanceof Comparable && p2.secondComponent instanceof Comparable) {
				return ((Comparable)p1.secondComponent).compareTo(p2.secondComponent);
			}
			return 0;
		}
	}
	
	private Comparator c;
	private ArrayList pairs = new ArrayList();
}

