package econet;

import java.util.Collection;
import java.util.HashMap;
import java.util.Vector;

import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.dom.IMethodBinding;
import org.eclipse.jdt.core.dom.IVariableBinding;



/**
 * Table for storing informations about the types.
 * @author jroyer
  */
public class TypesTable extends  HashMap<String, Information>{

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Constructor.
	 * @param vt
	 */
	public TypesTable(Vector<IType> vt) {
		super();
		for (int i = 0; i < vt.size(); i++) {
			this.put(vt.get(i).getFullyQualifiedName(), new Information());
		}
	}
	
	/**
	 * Check the existence of such a type.
	 * @param fn
	 * @return
	 */
	public boolean isOfInterest(String fn) {
		return this.containsKey(fn);
	}
	
	/**
	 * Check if it is flagged as a data type.
	 *  Require this.isOfInterest()
	 * @param fn
	 * @return
	 */
	public boolean isDataType(String fn) {
		return this.get(fn).isDataType();
	}

	/**
	 * Check if the type name is resolved
	 * Require this.isOfInterest()
	 * @param fn
	 * @return
	 */
	public boolean isResolved(String fn) {
		return (this.get(fn).isResolved());
	}
	
	/**
	 * Test fields existence.
	 * @param fn
	 * @return
	 */
	public boolean hasFields(String fn) {
		return this.get(fn).hasFields();
	}
	
	/**
	 * Set that name is a data type.
	 * @param bn binary name
	 */
	public void setDataType(String bn) {
		if (this.containsKey(bn)) {
			Information inf = this.get(bn);
			inf.setDataType(true);
			this.put(bn, inf);
			propagateDATA();

		}
	}
	
	/**
	 * Set that name is a component type.
	 * @param bn binary name
	 */

	public void setComponentType(String bn) {
		if (this.containsKey(bn)) {
			Information inf = this.get(bn);
			inf.setComponentType(true);
			if (inf.isDataType()) {
				this.put(bn, inf);
				propagateDATA();
			}
		}
	}
	
	/**
	 * Set as THE root name. 
	 * @param bn
	 */
	public void setRoot(String bn) {
		if (this.containsKey(bn)) {
			Information inf = this.get(bn);
			inf.setRoot();
			this.put(bn, inf);
		}
	}
	
	/**
	 * Add a new entry and its information.
	 * @param fn
	 * @param it
	 */
	public void addType(String fn, Information it) {
		this.put(fn, it);
	}
	
	/**
	 * Set the information type for a name.
	 * @param bn
	 * @param it
	 */
	public void setType(String bn, MyIType it) {
		if (this.containsKey(bn)) {
			Information inf = this.get(bn);
			inf.setType(it);
			this.put(bn, inf);
		} 
	}
	
	/**
	 * test if sub inherits directly or not from sup
	 * IF sup is external we consider it is true
	 * @param sub full name of type in the table
	 * @param sup full name of any type
	 * @return
	 */
	public boolean inheritsFromData(String sub, String sup) {
		boolean result = false;
		if (!sub.equals(sup) && this.isOfInterest(sub)) {
			if (!this.isOfInterest(sup)) {
				result = true;
			} else {
				// TODO attention ibind != null
				String thesup = this.get(sub).getIbind().getSupername();
				result = thesup.equals(sup) || this.inheritsFromData(thesup, sup);
			}
		}
		return result;
	}
	
	/**
	 * For each entry flagged DATA subclasses are also
	 * flagged as DATA.
	 */
	public void propagateDATA() {
		for (String key : this.keySet()) {
			if (this.get(key).isDataType()) {
				for (String subc : this.get(key).getIbind().getSubclasses()) {
					this.get(subc).setDataType(true);
				}
			}
			else {
				if (this.get(key).isComponentType()) {
					for (String subc : this.get(key).getIbind().getSubclasses()) {
						this.get(subc).setComponentType(true);
					}
				}
			}
		}
	}

	
	/**
	 * Compute the list of component which have a public fields
	 * @return
	 */
	public Vector<String> checkPublicFields() {
		Vector<String> res = new Vector<String> ();
		for (String key : this.keySet()) {
			if (this.get(key).isComponentType() && this.get(key).getIbind().hasPublicFields()) {
				res.add(key);
			}
		}
		return res;
	}
	
	/**
	 * Compute the component type which have a cyclic structure.
	 * Look only at component type and catch all the connected components
	 * @return
	 */
	public Vector<String> checkCycle() {
		// set initial flag for DATA type
		for (String key : this.keySet()) {
			if (!this.get(key).isComponentType()) {
				this.get(key).setViewed(true);
			}
		}
		// look the component type not already viewed
		// to explore all the connected component of the graph structure
		Vector<String> res = new Vector<String> ();
		for (String key : this.keySet()) {
			if (!this.get(key).isViewed()) {
				//System.out.println(" relance pour " + key);
				res.addAll(this.findCycle(key, new Vector<String> ()));
			}
		}
		return res;
	}
	
	/**
	 * Recursive function to find cyclic component type.
	 * @param node
	 * @param history
	 * @return
	 */
	public Vector<String> findCycle(String node, Vector<String> history) {
		Vector<String> res = new Vector<String> ();
		//System.out.println(" visit " + node + " h = " + history);
		Information info = this.get(node);
		// from structure construction these are component types
		if (history.contains(node)) {
				res.add(node);
			info.setCycle(true);
		} else { 
			// look at the fields and explore
			for (IVariableBinding f : info.getIbind().getStructure()) {
				String fname = f.getType().getQualifiedName();
				Information inff = this.get(fname);
				// already visited
				if (inff.isViewed()) {
					if (inff.isCycle()) {
						// forget multiple occurences
						if (!res.contains(node)) {
							res.add(node);
						}
						info.setCycle(true);
					}
				} else {
					history.add(node);
					// forget multiple occurences
					for (String st : this.findCycle(fname, history)) {
						if (!res.contains(st)) {
							res.add(st);
						}
					}
				}	
			}
		}
		// flag node as viewed
		info.setViewed(true);
		return res;
	}
	
	/**
	 * Setting default null information.
	 * @param bn
	 */
	public void setType(String bn) {
			this.put(bn, new Information());
	}
	
	/**
	 * Get the structure stored in the table.
	 * @param fullname
	 * @return
	 */
	public Fields getStructure(String fullname) {
		return this.get(fullname).getIbind().getStructure();
	}
	
	/**
	 * String description.
	 */
	public String toString() {
		String result = "";
		for (String key : this.keySet()) {
			result += key  + " -> " + this.get(key) + "\n";
		}
		return result;
	}
	
	public boolean isComponentType(String fn) {
		return (this.get(fn).isComponentType());
	}
	
	/**
	 * add a method into the provided interface of a component
	 * @param fn name of the component
	 * @param method method to add to the component provided interface
	 * @throws Exception it must be a component
	 */
	public void addProvidedMethod(String fn, IMethodBinding method) throws Exception {
		this.get(fn).addProvidedMethod(method);
	}
	
	/**
	 * add a method into the required interface of a component
	 * @param fn name of the component
	 * @param method method to add to the component required interface
	 * @throws Exception it must be a component
	 */
	public void addRequiredMethod(String fn, IMethodBinding method) throws Exception {
		this.get(fn).addRequiredMethod(method);
	}
	
	/**
	 * add methods into the provided interface of a component
	 * @param fn name of the component
	 * @param method methods to add to the component provided interface
	 * @throws Exception it must be a component
	 */
	public void addProvidedMethods(String fn, Collection<IMethodBinding> methods) throws Exception {
		this.get(fn).addProvidedMethods(methods);
	}
	
	/**
	 * add methods into the required interface of a component
	 * @param fn name of the component
	 * @param method methods to add to the component required interface
	 * @throws Exception it must be a component
	 */
	public void addRequiredMethods(String fn, Collection<IMethodBinding> methods) throws Exception {
		this.get(fn).addRequiredMethods(methods);
	}
	
	/**
	 * get the required interface of the component
	 * @param fn name of the component
	 * @return the required interface
	 * @throws Exception it must be a component
	 */
	public Interface getRequiredInterface(String fn) throws Exception {
		return this.get(fn).getRequiredInterface();
	}
	
	/**
	 * get the provided interface of the component
	 * @param fn name of the component
	 * @return the provided interface
	 * @throws Exception it must be a component
	 */
	public Interface getProvidedInterface(String fn) throws Exception {
		return this.get(fn).getProvidedInterface();
	}
	
	
	/**
	 * set the required interface of the component
	 * @param fn name of the component
	 * @return the required interface
	 * @throws Exception it must be a component
	 */
	public void setRequiredInterface(String fn, Interface requiredInterface) throws Exception {
		this.get(fn).setRequiredInterface(requiredInterface);
	}
	
	/**
	 * set the provided interface of the component
	 * @param fn name of the component
	 * @throws Exception it must be a component
	 */
	public void setProvidedInterface(String fn, Interface requiredInterface) throws Exception {
		this.get(fn).setProvidedInterface(requiredInterface);
	}
	


}
